#!/usr/bin/perl -w
#
# install Netop Guest or Netop Host rpm / tar.gz found in current directory.
# (c) Netop Business Solutions A/S 2016
#

my $VERSION = "install: 1.08";
my $DEBUG = 0;
my $VERBOSE = 1;

use Cwd;
use File::Path;
use File::Copy;
use File::Basename;

use lib '.';

require "tarinst.pm";
require "rpminst.pm";
require "yuminst.pm";
require "zypinst.pm";
require "debinst.pm";

my $PERL_CMD        = "/usr/bin/perl";
my $GZCAT_CMD       = "/usr/bin/gzcat";
my $CP_CMD          = "/bin/cp";
my $MV_CMD          = "/bin/mv";
my $LS_CMD          = "/bin/ls";
my $RPM_CMD         = "/bin/rpm";
my $ID_CMD          = "/usr/bin/id";
my $YUM_CMD         = "/usr/bin/yum";
my $ZYP_CMD         = "/usr/bin/zypper";
my $TAR_CMD         = "/bin/tar";
my $APT_CMD         = "/usr/bin/apt-get";
my $DPKG_CMD         = "/usr/bin/dpkg";
my $TAR_EXTRACT     = "zxf";
my $TAR_TOC         = "ztvf";
my $TAR_EXTRACT_CMD = "";
my $TAR_TOC_CMD     = "";
my $TAR_EXTENSION   = ".tar.gz";
my $SKIP=`awk '/^__LICENSE_FOLLOWS__/ { printf("%d", NR + 1); exit 0; }' $0`;
my $CRTDIR=cwd();
my $THIS="$0";
my $NETOPGUEST_BIN = "netopguest";
my $NETOPGUEST = "/opt/netop/guest/bin/" . $NETOPGUEST_BIN;
# change product version here, it will be seen by each <package>inst.pm
my $NETOP_PRODUCT_VERSION = '12.79';
my $GUEST_LICENSE_DIR = "/var/opt/netop/guest";
my $GUEST_LICENSE_FILE = $GUEST_LICENSE_DIR."/guest.lic";

my $NETOPHOST_ROOT_DIR = "/opt/netop/host";
my $NETOPHOST_BIN = "netophost";
my $NETOPHOST = $NETOPHOST_ROOT_DIR."/bin/".$NETOPHOST_BIN;

my $NETOPHOST_VAR_ROOT_DIR = "/var/opt/netop/host";

my $HOST_INIFILE = "/var/opt/netop/host/host.xml";

sub ubuntu_10_x64
{
    my $version_file = "/etc/issue";
    my $version = 10;
    my $cpu = "x86_64";
    # this only works on Ubuntu ! next time test it on other distros !
	#return !system("[ \$(uname -m) = $cpu -a -f $version_file -a \$(head -n 1 $version_file |\
	#    cut -d \" \" -f 2 | cut -d \".\" -f 1) -eq $version ]");
	return !system("[ \$( uname -m ) = $cpu -a -f $version_file -a \$(head -n 1 $version_file |\
					cut -d \" \" -f 2 | cut -d \".\" -f 1) = $version ]");  
}

sub debian_6_x64
{
	return !system("[ \$(uname -m) = x86_64 -a -n \"\$(cat /etc/issue | grep Debian | grep 6)\" ]");
}

sub debian_7_x64
{
	return !system("[ \$(uname -m) = x86_64 -a -n \"\$(cat /etc/issue | grep Debian | grep 7)\" ]");
}

sub prepare_debian_6_x64
{
    system("$APT_CMD -y install ia32-libs");
}

sub prepare_debian_7_x64
{
    system("$DPKG_CMD --add-architecture i386; $APT_CMD update; $APT_CMD -y install libc6:i386");
}

sub prepare_debian()
{
    if (debian_6_x64) {
        prepare_debian_6_x64();
    }

    if (debian_7_x64) {
        prepare_debian_7_x64();
    }
}

sub install_libstdcpp_5
{
   my $sLibName = "libstdc++.so.5";
   my $bFoundLib = 0;
   my @list = ("/usr/lib", "/lib", "/usr/local/lib");
   my $sFound = "";
   if(`uname` =~ "Linux") {
    return;
   }
   
   
   while(my $libpath = shift(@list)) {
      if ( $sFound eq "" ) {
         $sFound    = `find $libpath -name $sLibName -print`;
      }
      chomp $sFound;
   }

   if($sFound eq "" && `uname` =~ "SunOS") {
      $sFound   = `find "/usr/sfw/lib" -name $sLibName -print`;
      chomp $sFound;
   }
        
   if ( $sFound eq "" ) {
      print "Note: $sLibName not found trying to install... \n";
      my $rpm_cmd = "rpm -ivh compat-libstdc++-5.0.7-6.i586.rpm";
      system($rpm_cmd); 
   }
}

sub get_tool
{
    my ($arg) = @_;
    my $tool="";
    my $cmd = "unalias cp 2>/dev/null; which $arg 2>/dev/null";
    dbg("cmd: $cmd\n");
    $tool = `$cmd`;
    $tool =~ s/\n//g;
    if($tool =~ m|[ \t]|) {
       $tool = ""; # contained whitespace, must be a message.
    }

    return $tool;
}

sub dir_is_empty
{
    my $dir_name = shift;
    opendir(my $dir_handle, $dir_name) or return 0;
    my $entries_num = scalar(grep {$_ ne "." && $_ ne ".."} readdir($dir_handle));
    closedir($dir_handle);
    return ($entries_num == 0);
}


sub convert_host_files
{
    my $old_host_cfg_dir = "/etc/NetopHost";
    
    if ( -f "$old_host_cfg_dir/NetopHost.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
        system("$MV_CMD $old_host_cfg_dir/NetopHost.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
    }
    if ( -f "$old_host_cfg_dir/NetopHost.dwm" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.dwm") {
        system("$MV_CMD $old_host_cfg_dir/NetopHost.dwm $NETOPHOST_VAR_ROOT_DIR/host.dwm");
    }
    if (-f "$old_host_cfg_dir/host.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
        system("$MV_CMD $old_host_cfg_dir/host.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
    }
    if (-f "$old_host_cfg_dir/license.dat" && ! -f "$NETOPHOST_VAR_ROOT_DIR/license.dat") {
        system("$MV_CMD $old_host_cfg_dir/license.dat $NETOPHOST_VAR_ROOT_DIR/license.dat");
    }
    if (-f "$old_host_cfg_dir/host.lic" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.lic") {
        system("$MV_CMD $old_host_cfg_dir/host.lic $NETOPHOST_VAR_ROOT_DIR/host.lic");
    }
    if (-f "$old_host_cfg_dir/host.uuid" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid") {
        system("$MV_CMD $old_host_cfg_dir/host.uuid $NETOPHOST_VAR_ROOT_DIR/host.uuid");
    }
    if (-f "$old_host_cfg_dir/host.uuid_old" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid_old") {
        system("$MV_CMD $old_host_cfg_dir/host.uuid_old $NETOPHOST_VAR_ROOT_DIR/host.uuid_old");
    }
    
    if (-d $old_host_cfg_dir && dir_is_empty($old_host_cfg_dir)) {
        rmdir($old_host_cfg_dir);
    }
    
    if (-f "/etc/netop.ini" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.ini") {
        system("$CP_CMD /etc/netop.ini $NETOPHOST_VAR_ROOT_DIR/host.ini");
    }
}

sub setup_tools
{
    my $autoinstall = shift;
    if ( ! $autoinstall ) { $autoinstall = 0; }
    if(! -e $TAR_CMD) {
        $TAR_CMD = get_tool("tar");
        if(! -e $TAR_CMD ) {
            die "Non-standard tar, please update TAR_CMD\n";
        }
    }
    if(!-e $LS_CMD) {
        $LS_CMD = get_tool("ls");
        if(! -e $LS_CMD ) {
            die "Non-standard ls, please update LS_CMD\n";
        }
    }
    if(! -e $CP_CMD ) {
        $CP_CMD = get_tool("cp");
        if(! -e $CP_CMD ) {
            die "Non-standard cp, please update CP_CMD\n";
        }
    }
    if(! -e $MV_CMD ) {
        $MV_CMD = get_tool("mv");
        if(! -e $MV_CMD ) {
            die "Non-standard mv, please update MV_CMD\n";
        }
    }
    if(! -e $RPM_CMD) {
        $RPM_CMD = get_tool("rpm");
    }
    
    if(! -e $YUM_CMD) {
        $YUM_CMD = get_tool("yum");
    }
    
    if(! -e $ZYP_CMD) {
        $ZYP_CMD = get_tool("zypper");
    }
    
    if(! -e $APT_CMD) {
        $APT_CMD = get_tool("apt-get");
    }
    
    if(! -e $GZCAT_CMD) {
        $GZCAT_CMD = get_tool("gzcat");
    }
    if(! -e $ID_CMD) {
        $ID_CMD = get_tool("id");
    }
    $TAR_EXTRACT_CMD = "$TAR_CMD $TAR_EXTRACT";
    $TAR_TOC_CMD     = "$TAR_CMD $TAR_TOC";

    $ENV{"PERL_CMD"}        = $PERL_CMD;
    $ENV{"GZCAT_CMD"}       = $GZCAT_CMD;
    $ENV{"CP_CMD"}          = $CP_CMD;
    $ENV{"MV_CMD"}          = $MV_CMD;
    $ENV{"LS_CMD"}          = $LS_CMD;
    $ENV{"TAR_CMD"}         = $TAR_CMD;
    $ENV{"NETOP_PRODUCT_VERSION"} = $NETOP_PRODUCT_VERSION;

    # call each type of package installer with an argument to disable user questions :
    if ( $autoinstall )
    {
        $ENV{"AUTOINSTALL"}     = "y";
        $ENV{"RPM_CMD"}         = $RPM_CMD;  # rpm is non-interactive by default
        $ENV{"YUM_CMD"}         = $YUM_CMD . " -y "; # "--assumeyes" does not work with all yum-type package managers
        $ENV{"ZYP_CMD"}         = $ZYP_CMD . " --non-interactive ";
        $ENV{"APT_CMD"}         = $APT_CMD . " -y ";
        $ENV{"DPKG_CMD"}        = $DPKG_CMD;
    }
    else
    {
        $ENV{"AUTOINSTALL"}     = "n";
        $ENV{"RPM_CMD"}         = $RPM_CMD;
        $ENV{"YUM_CMD"}         = $YUM_CMD;
        # can't wrap zypper binary with a pipe ... 
        $ENV{"ZYP_CMD"}         = $ZYP_CMD . " --non-interactive ";
        $ENV{"APT_CMD"}         = $APT_CMD;
        $ENV{"DPKG_CMD"}        = $DPKG_CMD;
    }
    $ENV{"TAR_EXTRACT"}     = $TAR_EXTRACT;
    $ENV{"TAR_TOC"}         = $TAR_TOC;
    $ENV{"TAR_EXTRACT_CMD"} = $TAR_EXTRACT_CMD;
    $ENV{"TAR_TOC_CMD"}     = $TAR_TOC_CMD;
    $ENV{"TAR_EXTENSION"}   = $TAR_EXTENSION;
}

sub dbg
{
    if($DEBUG) {
        my $arg = shift(@_);
        print STDERR "DBG: $arg";
    }
}

sub input_confirmation
{
   my $options = "[Yes/no/quit]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if(($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } elsif (($lower =~ ("n"))) {
           return "n";
       } elsif (($lower =~ ("q"))) {
           return "q";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub su_test
{
    if(! -e $ID_CMD) {
        return 1; # continue, cannot get uid.
    }
    my $id = `$ID_CMD`;
    if($id =~ m|uid=([0-9]+)|) {
        if($1 eq "0") {
            return 1;
        }
    }
    return 0;
}

# $_[0] is product name to display (MANDATORY)
# $_[1] is the validator subroutine passed as a reference (MANDATORY) that must return a boolean flag.
# $_[2] is the patience ( times to ask ) (OPTIONAL) = 3
# $_[3] is an optional argument to the validator (OPTIONAL) = undef
sub patiently_ask_for_serial
{
    ( $product_name, $validator, $patience, $validator_arg ) = @_;
    $answer = undef;
    return ( 0, $answer ) unless ( ( defined $product_name ) and ( defined $validator ) );
    $patience = 1 * $patience; # force numeric context
    $patience = 3 unless ( ( defined $patience ) and ( $patience ) ); # default patience : perl is patient
    while ( $patience )
    {
        my $options = "[T]rial / [p]ortal / [c]ustom";

        print "\n[$patience] Which type of license do you want to use for $product_name? $options:\n";
        $answer = <STDIN>;
        
        my $lower = lc $answer;
        if ( ($lower eq "p\n"))
        { 
            $answer = "PORTAL"; 
        }
        elsif (($answer eq "\n") || ($lower eq "t\n")) 
        { 
            $answer = "TRIAL"; 
        }
        elsif (($lower eq "c\n")) 
        { 
            print "\n\t[$patience] Enter License key for $product_name ( or leave blank for trial ):\n";
            $answer = <STDIN>;

            if ( (!$answer) or ($answer =~ /^\s*$/) )  { $answer = "TRIAL"; } 
        }
        else
        {    
             print "Please reply: $options\n";
         next;  
        }

        if ( $validator->( $answer, $validator_arg ) ) { return ( $patience, $answer ); }
        $patience--;
        print "\nThe license key could not be validated, please try again. ($patience)\n";
    }
    print STDERR "\nNone of the license keys could be validated.\n";
    return ( 0, $answer );
} # sub patiently_ask_for_serial

#
# solaris crle config begin
#

$SYSTEM = "";
$OS_VER = "";
$OS_BIT = "";

# This script only checks whether SYSTEM is "solaris" or not.
# And OS_VER is only used for SYSTEM = "solaris".
sub system_setup {
    my $uname = `uname`;
    dbg("uname  = $uname");
    if ($uname =~ "SunOS") {
        $SYSTEM = "solaris";
        $OS_VER = `uname -r`;
    } elsif ($uname =~ "Darwin") {
        $SYSTEM = "Darwin";
        $OS_VER = `uname -r`;
    } elsif ($uname =~ "WindowsNT") {
        $SYSTEM = "WindowsNT";
        $OS_VER = "8.7";
    } else {
        $SYSTEM = "linux";
        $OS_VER = "";
        $OS_BIT = `uname -m`;
    }
    dbg("SYSTEM = $SYSTEM\n");
    dbg("OS_VER = $OS_VER\n");
}

my $CRLE_CONFIG  = "/var/ld/ld.config";
my $LDD_BINARY = "/bin/ldd";
my $CRLE_PATH = "";

sub configure_crle()
{
    return 1 if ($CRLE_PATH eq "");

    if (-f $CRLE_CONFIG) {
          # file exist, use update flag: -u
          $cmd = "crle -u -l $CRLE_PATH";
    } else {
          # file does not exist, remember to include /usr/lib
          $cmd = "crle -l /usr/lib:$CRLE_PATH";
    }
    dbg("configure_crle: crle_command: '$cmd'\n");
    if (system $cmd ) {
          die "error in $cmd\n";
    }
}

#
# BEGIN ldd test tools
#
$FILE_EXPR  = "";
$SEARCH_DIR = "";
# locate $FILE_EXPR in $SEARCH_DIR
# return 1 if match, 0 if no match
sub find_file()
{
    my @filelist = `/bin/ls $SEARCH_DIR`;
    while ($file = pop @filelist) {
       return 1 if($file =~ m|$FILE_EXPR| );
    }
    return 0;
}

sub append_crle_path()
{
    if($CRLE_PATH eq "") {
          $CRLE_PATH = $SEARCH_DIR;
    } else {
          if(!($CRLE_PATH =~ $SEARCH_DIR)) {
                $CRLE_PATH = "$CRLE_PATH:$SEARCH_DIR";
          }
    }
}

$SO_MISSING_LIST = "";
$MISSING = "";
sub append_so_missing_list()
{
    if($SO_MISSING_LIST eq "") {
          $SO_MISSING_LIST = $MISSING;
    } else {
          if(!($SO_MISSING_LIST =~ $MISSING)) {
                $SO_MISSING_LIST = "$SO_MISSING_LIST, $MISSING";
          }
    }
}

# build directory list for CRLE -> $CRLE_PATH.
# return 0 if all ok.
# return 1 if CRLE path needs updated $CRLE_PATH is updated.
# return 2 if unresolved libraries exist (needs to be installed on system).
sub test_ldd
{
    my $st=0;
    my ($exe) = @_;
    my @ld = `$LDD_BINARY $exe`;
    while ($line = pop @ld) {
          my $WSP = "[ \t]";
          my $NWSP = "[^ \t]";
          my $regexp = "^$WSP*(lib$NWSP+.so$NWSP*)$WSP+=>$WSP+[(][^)]+[)].*\$";
          if($line =~ m|$regexp| ) {
                $missing = $1;
                if($missing =~ m|libXtst.so| ) {
                     $SEARCH_DIR = "/usr/openwin/lib";
                     $FILE_EXPR  = "libXtst.so";
                     if(find_file() == 1) {
                          if ($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
                          $st = 1 if($st < 1);
                          append_crle_path();
                     } else {
                          if ($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
                          $MISSING = $missing;
                          append_so_missing_list();
                          $st = 2 if($st < 2);
                     }
                } elsif ($missing =~ m|libstdc[+][+].so| ) {
                     $SEARCH_DIR = "/usr/local/lib";
                     $FILE_EXPR  = "libstdc[+][+].so";
                     if(find_file() == 1) {
                          if ($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
                          $st = 1 if($st < 1);
                          append_crle_path();
                     } else {
                          if ($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
                          $MISSING = $missing;
                          append_so_missing_list();
                          $st = 2 if($st < 2);
                     }
                } elsif ($missing =~ m|libgcc| ) {
                     $SEARCH_DIR = "/usr/local/lib";
                     $FILE_EXPR  = "libgcc";
                     if (find_file() == 1) {
                          if ($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
                          $st = 1 if($st < 1);
                          append_crle_path();
                     } else {
                          if ($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
                          $MISSING = $missing;
                          append_so_missing_list();
                          $st = 2 if($st < 2);
                     }
                } else {
                     if($VERBOSE) {printf "Missing ref to $1 cannot resolve.\n"};
                     append_so_missing_list();
                     $MISSING = $missing;
                     $st = 2 if($st < 2);
                }
          }
    }
    return $st;
}
#
# END ldd test tools
#

#
# Forall executable: ensure that it can be run.
#
sub verify_requirements
{
    my $fatal_missing=0;
    my @exe_filelist = @_;
    while(my $exe = pop @exe_filelist) {
        die "Missing executable: '$exe'\n" if(! -e $exe);
        my $st = test_ldd($exe);
          if($st > $fatal_missing)  { $fatal_missing = 1};
          if($st > $fatal_missing)  { $fatal_missing = 2};
    }

    return $fatal_missing;
}

my $NEED_LD_PATH=0;
sub ensure_solaris_environment
{
    my $exe="";
    if($SYSTEM ne "solaris") {
        return 1;
    }
    my ($type) = @_;
    if($type eq "guest") {
        $exe = "/usr/bin/netopguest"
    } elsif ($type eq "host") {
        $exe = "/usr/sbin/netophost"
    } else {
        dbg("ensure_solaris_environment: Illegal argument: '$type'\n");
        return 0;
    }
    dbg("ensure_solaris_environment: type: '$type'\n");
    dbg("ensure_solaris_environment: checking executable: '$exe'\n");

    my $ld_library_path = $ENV{"LD_LIBRARY_PATH"};
    my $st = verify_requirements($exe);
    dbg("CRLE_PATH = '$CRLE_PATH'\n");
    # append LD_LIBRARY_PATH to crle if exist.
    if($st == 2) {
         die "Missing dynamic libraries on system: $SO_MISSING_LIST";
    }
    if ($st != 0 || ($CRLE_PATH ne "")) {
        configure_crle();
    }
    if (verify_requirements($exe)) {
        my $VER="8";
        if ($OS_VER eq "5.9") {
            $VER = "9";
        }
        if (! -d "/usr/local/lib" || ! -e "/usr/local/lib/libstdc++.so.5") {
            print "! \n";
            print "! Warning: required library: libstdc++.so.5 is not installed on the system.\n";
            print "! libstdc++.so.5 is located in package SMClibgcc.\n";
            print "! SMClibgcc can be downloaded from www.sunfreeware.com\n";
            print "! or from any of the mirror sites.\n";
            print "! (Download the file: libgcc-3.2.X-sol$VER-sparc-local)\n";
            print "! \n";
            print "! Install using pgkadd:\n";
            print "! pkgadd -d libgcc-3.2-sol$VER-sparc-local\n";
            print "! \n";
            return 0;
        }
        return 0;
    }
    return 1;
}
#
# solaris crle config end
#

#my $COPYRIGHT=
 
sub write_licfile
{
   my ($text,$licfile) = @_;
   open OUTFILE,">$licfile" or die "Cannot open '$licfile' for writing: $!\n";
   printf OUTFILE "%s",$text;
   close OUTFILE;
   print "License agreement written to $licfile\n\n";
}

sub input_confirmation_N
{
     my $options = "[yes/No/quit]";

     my $question = shift(@_);
     print "$question $options >";
     while(<STDIN>) {
          my $lower = lc $_;
          if( ($lower =~ ("y"))) {
                return "y";
          } elsif (($_ eq "\n") || ($lower =~ ("n"))) {
                return "n";
          } elsif (($lower =~ ("q"))) {
                return "q";
          } else {
                print "Please reply: $options\n";
                print "$question $options >";
          }
     }
}

# moved validation here :
# expects serial number as an argument 
sub validate_guest_serial_number
{
    my $serial = shift(@_);
    return 0 unless defined $serial;
	my $command = "$NETOPGUEST -no_splash -serialno $serial";
	my $status = system($command);
    if ( 0 == $status )
    {
        print "\n\tSerial Number validated OK.\n";
        return 1;
    }
    else
    {
        print STDERR "\nSerial Number is INVALID.\n";
        return 0;
    }
} # sub validate_guest_serial_number

sub guest_installed_but_not_validated
{
    if (-f $GUEST_LICENSE_FILE && -s $GUEST_LICENSE_FILE) {
        print "\nThe Netop Guest application has been installed.\n";
        print "Netop Guest will try to use the existig license file ($GUEST_LICENSE_FILE)\n";
    } else {
        print "\nThe Netop Guest application has been installed, but not validated.\n";
        print "Please run '$NETOPGUEST_BIN -serialno <your license key>' as super-user to validate the product.\n";
        print "Afterwards, you can start the Netop Guest application by typing '$NETOPGUEST_BIN' as a regular user.\n";
    }
    print "Thank you for using Netop Solutions!\n\n";
} # sub guest_installed_but_not_validated

sub guest_installed_and_validated
{
    print "\nNetop Guest application has been installed and activated.\n";
    print "You can start the Netop Guest application by typing '$NETOPGUEST_BIN' as a regular user.\n";
    print "Thank you for using Netop Solutions!\n\n";  
} # sub guest_installed_and_validated

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license
{
    my $scroll_method = "more";
    my $allow_scroll = shift;
    $allow_scroll = 0 + $allow_scroll; # force numeric ctx
    if ( ! defined $allow_scroll ) { $allow_scroll = 1; }
    my $command = "tail -n +$SKIP $THIS";
    if ( 1 == $allow_scroll ) { $command = $command . " | $scroll_method"; }
    system($command);
} # sub print_license

# 1st argument is a reference (!)
sub select_product_question
{
    ( $product_ref, $product_name ) = @_;
    my $error = "\nInvalid arguments in : sub select_product_question\n";
    goto return_error unless ( ( defined $product_ref ) and ( defined $product_name) );
    my $answer = input_confirmation("Install " . $product_name . " <Yes/no/quit> ? ");
    $answer = lc substr( $answer, 0, 1 ); # allow both 'yes' and 'y' answers
    if ( 'q' eq $answer ) { print "Aborting instalation on user request.\n"; exit 1; } # EXIT POINT (!)
    if ( 'y' eq $answer ) { $$product_ref = 1; print "You will be prompted for $product_name license key during instalation.\n"; }
    return $$product_ref;
return_error:
    print STDERR "$error";
    return undef;
} # sub select_product_question

#checks if the netophost/guest package exists
#params are : dir and file name to check
sub check_file_exists
{
  my $dir= shift(@_);
  my $filech = shift(@_);

  opendir(DIR, $dir) or die $!;

  while (my $file = readdir(DIR)) {
    next if ($file !~ m/^$filech.*(deb$|rpm$|tar\.gz$)/);
    return 1;
  }

  closedir(DIR);
  return 0;
}

sub migrate_guest_license()
{
    if (-f "/etc/NetopGuest/guest.lic" && ! -f "$GUEST_LICENSE_FILE") {
        mkpath($GUEST_LICENSE_DIR) unless (-e $GUEST_LICENSE_DIR);
        system("$MV_CMD /etc/NetopGuest/guest.lic $GUEST_LICENSE_FILE");
    }
    if (-d "/etc/NetopGuest/" && dir_is_empty("/etc/NetopGuest/")) {
        rmdir("/etc/NetopGuest/");
    }
}

sub main()
{
    my $host  = 0;
    if (defined $DEFAULT_HOST) {
        $host = $DEFAULT_HOST;
    }
    
    my $guest = 0;
    my $norpm       = 0;
    my $nodeb       = 0;
    my $autoinstall_flag = "--autoinstall";
        
    my $autoinstall = 0;
    if (defined $DEFAULT_AUTOINSTALL) {
        $autoinstall = $DEFAULT_AUTOINSTALL;
    }
    
    my $guest_serial = "";
    my $host_serial = "";
    my $oldserial   = "";
    my $usage =  "Usage: install.pl <param>\n"
                ." where <param> is:\n"
                ."    [--help]           : print help message and exit.\n"
                ."    [--version]        : print version info and exit.\n"
                ."    [--guest <serial>] : install Netop Guest with <serial> number license.\n"
                ."    [--host <serial>]  : install Netop Host with <serial> number license.\n"
                ."    [--norpm]          : use tar.gz install instead of 'rpm'.\n"
                ."    [--debug]          : turn debug on.\n"
                ."    [--license]        : print the Netop License.\n"
                ."    [--autoinstall]    : non-interactive install, assumes you agree with the Netop License.\n"
                ."    [--oldserial <serial>] : old license serial for host, if upgrading.\n";

    
    while($arg = shift @ARGV) {
        if ($arg eq "--version") {
            print "$VERSION\n";
            exit;
        } elsif ( $arg eq "--license" ) {
            print_license( 1 );
            exit ( 0 );
        } elsif ($arg eq "--debug") {
            $DEBUG = 1;
        } elsif ($arg eq "--autoinstall") {
            $autoinstall_flag = $arg;
            $autoinstall = 1;
        } elsif ($arg eq "--norpm") {
            $norpm = 1;
        } elsif ($arg eq "--help") {
            print $usage;
            exit;
        } elsif ($arg eq "--host") {
            $host = 1;
            # next token is not another option ?
            if ( ( defined $ARGV[0] ) and ( "--" ne substr($ARGV[0],0,2) ) ) { $host_serial = shift @ARGV; } #
        } elsif ($arg eq "--guest") {
            $guest = 1;
            # next token is not another option ?
            if ( ( defined $ARGV[0] ) and ( "--" ne substr($ARGV[0],0,2)) ) { $guest_serial = shift @ARGV; } #
        } elsif ($arg eq "--oldserial") {
            my $num = shift @ARGV;
            $oldserial = "--serial $num";
          } else {
                print "Error: unknown arg: '$arg'\n";
            die $usage;
          }
    }
    
    die "Must have super-user privileges to install this application.\nPlease elevate your privileges with 'su -' and try again.\n" if(! su_test());

    install_libstdcpp_5();
    setup_tools( $autoinstall );
    if($RPM_CMD eq "") {
        $norpm = 1;
    }
# determine if rpm-files available
    {
        my $rpmfiles = `$LS_CMD Netop*.rpm 2>/dev/null`;
        $rpmfiles =~ s/\n//g;
        if($rpmfiles eq "") {
            $norpm = 1;
        }
    }
# determine if deb-files available
    {
        my $debfiles = `$LS_CMD Netop*.deb 2>/dev/null`;
        $debfiles =~ s/\n//g;
        if($debfiles eq "") {
            $nodeb = 1;
        }
    }

# must specify guest or host.
    if ( $autoinstall )
    {
        die "Error : --autoinstall flag requiers you provide at least --host <serial> or --guest <serial> ( or both )."
        if ( !$guest and !$host );
    }
    else
    {
        my $product_selected = 0;
        if ( 1 == $host ) # --host argument in command line
        {
            $product_selected = 1;
            my $message = "";
            $message = " with serial : $host_serial" unless ( "" eq $host_serial );
            print "Netop Host will be installed from --host option $message.\n";
        }
        if ( 1 == $guest ) # --guest argument in command line
        {
            $product_selected = 1;
            my $message = "";
            $message = " with serial : $guest_serial" unless ( "" eq $guest_serial );
            print "Netop Guest will be installed from --guest option $message.\n";
        }
        if ( 0 == $product_selected ) # no product selected in command line, ask user :
        {
            my $hpackage = check_file_exists( './', 'NetopHost');
            if( $hpackage == 1 ) {
                die "Installer Internal error.\n" unless defined select_product_question( \$host, "Netop Host" );
            }
            my $gpackage = check_file_exists( './', 'NetopGuest');
            if( $gpackage == 1 ) {
                die "Installer Internal error.\n" unless defined select_product_question( \$guest, "Netop Guest" );
            }
        }
    }
    
    die "None selected.\n" unless 1 == $host or 1 == $guest;
    
    dbg("Install Host  = $host\n");
    dbg("Install Guest = $guest\n");
    
    my $install_dir = `pwd`;
    $install_dir =~ s/\n//g;

    system_setup();

    if($SYSTEM eq "solaris") {
        setup_tools();
    }
    
    my $deb_ca_cert = "/etc/ssl/certs/ca-certificates.crt";
    my $rh_ca_cert = "/etc/pki/tls/certs/ca-bundle.crt";
    my $sles_ca_cert = "/etc/ssl/ca-bundle.pem";
    
    my $default_ca_cert = "ca-certificates.crt";       
    my $ca_cert_dir = "/var/opt/netop/certs/";
    my $ca_cert_file = "/var/opt/netop/certs/ca-bundle.crt";    
    
    #make sure the ca certificate is available in certificates directory.
    print "Installing ca certificates bundle in $ca_cert_dir";
    
    unless (-e $ca_cert_file) {
        eval { mkpath($ca_cert_dir) unless (-e $ca_cert_dir) };
        if ($@) {
            print "Couldn't create $ca_cert_dir: $@";
        }
            
        if (-e $deb_ca_cert) {
            symlink($deb_ca_cert, $ca_cert_file);
        }
        elsif (-e $rh_ca_cert) {
            symlink($rh_ca_cert, $ca_cert_file);
        }
        elsif (-e $sles_ca_cert) {
            symlink($sles_ca_cert, $ca_cert_file);
        }
        else {
            die "No ca certificate" unless (-e $default_ca_cert);        
            copy($default_ca_cert,  $ca_cert_file);
        }    
    }

    
install_guest:
    if ($guest) {
        print "\nInstalling Netop Guest $NETOP_PRODUCT_VERSION\n";
        # need to make a distinction between install failure and user
        # choosing N(o) in interactive CLI :
        my $result = 0;

        if (!$norpm)
        {
    	    if ($YUM_CMD) { $result = yuminst::install_guest($install_dir) }
            else {
                if($ZYP_CMD) { $result = zypinst::install_guest($install_dir); }
                else { $result = rpminst::install_guest($install_dir); }
    	    }
        }
        else
        {
            if ( !$nodeb and $APT_CMD and $DPKG_CMD)
            {
                prepare_debian();
                if (ubuntu_10_x64 or debian_6_x64) {
                    $result = debinst::install_guestx64($install_dir);
                } else {
                    $result = debinst::install_guestx32($install_dir);
                }
            } 
            else { $result = tarinst::install_guest($install_dir); }
        }
        # obsolete :
        if ($SYSTEM eq "solaris")
        {
               if (!ensure_solaris_environment("guest")) {
                   if (!ensure_solaris_environment("guest")) {  exit 1;  }
                }
        }
        # installation failed ?
        if ( 1 != $result )
        {
            # 'hard' failure of installer :
            if ( 0 == $result )
            {
                print STDERR "\nInstallation FAILED!.\n";
                goto install_host;
            }
            # user chose N(o) (or C(ancel)) in package manager :
            if ( 2 == $result )
            {
                print STDERR "You chose not to install this package on this system.\n";
                print STDERR "Installer can not continue for this product.\n";
                goto install_host;
            }
        }
        
        migrate_guest_license();
        
        print "\nThe following license-agreement apply to Netop Guest:\n";
        print "[Please read carefully]\n";
        my $res = '';
        # FIX : DE8512
        if ( 0 == $autoinstall )
        {
            print_license( ! $autoinstall );
            $res = input_confirmation_N("Do you accept this license ?");
            $res = substr($res,0,1);
            if ( 'y' ne $res )
            {
                print "License Agreement not accepted, program will not be installed!\n";
                goto install_host;
            }
        }
        else
        {
            print_license( ! $autoinstall );
            print "Auto-accepted license because of --autoinstall flag.\n";
        }
        if( -e "license.dat")
        {
            my $license = `cat license.dat`;
            print "The following trial license key apply to Netop Guest:\n";
            print "$license\n\n";
        }
    if ( "" eq $guest_serial )
    {
        if ( !$autoinstall )
        {
            ( $validated, $serial ) = patiently_ask_for_serial("Netop Guest", \&validate_guest_serial_number, 3, undef );
            if ( $validated )
            {
                if (0) { print "\t Validated with serial : $serial\n"; }
                guest_installed_and_validated();
                goto install_host;
            }
        }
        guest_installed_but_not_validated();
        goto install_host;
    } # if ( !$guest_serial )
    else
    {
        my $result = validate_guest_serial_number( $guest_serial );
        if ( 1 == $result )
        {
           guest_installed_and_validated();
           goto install_host;
        }
        guest_installed_but_not_validated();
    } # else ( !$guest_serial )

	} # if ($guest)
install_host:
    if ($host)
    {
        # check if the configuration file already exists
        my $config_file_exists = 0;
        if (-e $HOST_INIFILE) {
            $config_file_exists = 1;
        }

        print "\nInstalling Netop Host $NETOP_PRODUCT_VERSION\n";
        my $result = 0;
        
        if (!$norpm)
        {
            #print "Install RPM\n";
            if ($YUM_CMD){ $result = yuminst::install_host($install_dir); }
            else
            {
                if ($ZYP_CMD) { $result = zypinst::install_host($install_dir); }
                else { $result = rpminst::install_host($install_dir); }
            }
        }
        else
        {
            if (!$nodeb && $APT_CMD && $DPKG_CMD)
            {
                #print("Install DEB\n");
                prepare_debian();
                if (ubuntu_10_x64 or debian_6_x64) {
                    debinst::install_pammodules64($install_dir);
                    $result = debinst::install_hostx64($install_dir);
                } else {
                    if ($OS_BIT =~ "arm") {
                        $result = debinst::install_hostarmhf($install_dir);
                    } else {
                        $result = debinst::install_hostx32($install_dir);
                    }
                }
            }
            else { $result = tarinst::install_host($install_dir); }
        } # else norpm
        # obsolete :
        if ($SYSTEM eq "solaris")
        {
            if (!ensure_solaris_environment("host")) {
                if (!ensure_solaris_environment("host")) { exit 1; }
                }
        } # if solaris

        # installation OK :
        if ( 1 != $result )
        {
            # 'hard' failure of installer :
            if ( 0 == $result )
            {
                print STDERR "\nInstallation FAILED!.\n";
                exit( 1 );
            }
            # user chose N(o) in gdebi interactive install :
            if ( 2 == $result )
            {
                print STDERR "You chose not to install this package on this system.\n";
                print STDERR "Installer can not continue for this product.\n";
                exit( 1 );
            }
        }
        
        convert_host_files();
        
        my $configure_script = "/opt/netop/host/configure/configure.pl";
        if (! -e $configure_script) {
            print "Netop Host installed but not yet configured for use\n";
            exit;
        }

          if ( 0 == $autoinstall ) { $autoinstall_flag=""; }
          my $cmd = "perl $configure_script --serial $host_serial $oldserial $autoinstall_flag";
          if ($config_file_exists == 1) {
            $cmd = $cmd . " --upgrade"
          }
          dbg("cmd: $cmd\n");
          my $st = system($cmd);
          $st >>= 8;
          dbg("configure.pl st: $st\n");
          if ($st == 0) {
          } else {
                if ($st == 56) {
                     if (!$norpm) {
                          $method = "rpm -e NetopHost";
                     } else {
                          $method = "/opt/netop/host/configure/uninstall.pl";
                     }
                     print "\t(run '$method' to uninstall Netop Host)\n"; # TODO
                }
          }
    }
}

main;
exit;
__END__
__LICENSE_FOLLOWS__
Netop Business Solutions A/S End-User License Agreement 
Updated: April 2016

This Agreement provides you with the right to install, load, host and use “the Software” as described herein.  

Portions of the Software licensed under this Agreement are used under license from third parties. All Rights Reserved. 

NOTICE TO ALL USERS: PLEASE READ THE FOLLOWING LEGAL AGREEMENT ("AGREEMENT") CAREFULLY. THIS AGREEMENT CONSTITUTES A LEGAL, BINDING AND ENFORCEABLE AGREEMENT BETWEEN YOU AND NETOP BUSINESS SOLUTIONS A/S (“NETOP BUSINESS SOLUTIONS”) REGARDING ALL USE OF ANY NETOP BUSINESS SOLUTIONS SOFTWARE ("SOFTWARE"). BY INSTALLING THE SOFTWARE, YOU (EITHER AN INDIVIDUAL OR A SINGLE ENTITY) ACCEPT TO BE BOUND BY AND BECOME A PARTY TO THIS AGREEMENT WITH NETOP BUSINESS SOLUTIONS. IF YOU DO NOT AGREE TO ALL OF THESE TERMS, DO NOT INSTALL OR USE THE SOFTWARE. 

1. License Grant. Subject to the payment of the applicable license fees, and subject to your abidance by the terms and conditions of this Agreement, Netop Business Solutions hereby grants to you a non-exclusive, non-transferable right to use one copy of the specified version of the Software which you have acquired and the accompanying documentation (the "Documentation"). 

2.  Use. The use of products is described in appendices which is an integrated part of this End User Licenses Agreement and thus legally binding on You. Please find the proper appendix according to your product type which is stated on your invoice or order confirmation.  

3. Term. This Agreement is effective until terminated. You may terminate it at any point by destroying or deleting the Software together with all copies of the Software. Also, Netop Business Solutions has the option to terminate this Agreement if you fail to comply with any terms of this Agreement. You agree upon such termination immediately to destroy or deleting all parts of the Software and the Documentation together with all copies thereof. 

4. Updates. Netop Business Solutions may update the content of the Software from time to time and provided that you have purchased the right to upgrades, you may download such revisions, upgrades, or updates to the Software when and as Netop Business Solutions publishes them via its website or other online services. You agree that the terms of this Agreement (or any subsequent version hereof) shall applies to any updated version of the Software or the Documentation. 

5. Ownership Rights. International copyright laws and international treaty provisions protect the Software. Netop Business Solutions and its suppliers own and retain all right, title and interest in and to the Software and the Documentation, including all copyrights, patents, trade secret rights, trademarks and other intellectual property rights therein. Your possession, installation, or use of the Software does not transfer to you any title to the intellectual property in the Software, and you will not acquire any rights to the Software except as expressly set forth in this Agreement. All copies of the Software and Documentation made hereunder will contain the same proprietary notices that appear on and in the Software and the Documentation. 


6. Restrictions. You may not give away, rent, lease, loan or resell the Software and the Documentation or transfer any of the rights granted to you under this Agreement. You may not reverse engineer, decompile, hack, translate or disassemble the Software (except to the exact extent that applicable law expressly prohibits the foregoing restriction). You may not modify, or create derivative works based upon the Software in whole or in part. You may not copy the Software or Documentation except as expressly permitted in Section 1 above. You may not remove any proprietary notices or labels on the Software or Documentation. Nor may you cause or permit others to transact any of the actions which are governed by this Section 6. Netop Business Solutions reserves all rights not expressly set forth hereunder. 

7. Warranty and Disclaimer. 

a. Limited Warranty. Netop Business Solutions warrants that for sixty (60) days from the date of original purchase, the media (for example, the diskettes) on which the Software is contained will be free from defects in materials and workmanship.

b. Customer Remedies. Netop Business Solutions and its suppliers' entire liability, and your exclusive remedy for any breach of the foregoing warranty, shall be, at Netop Business Solutions' option, either (i) to return the purchase price paid for the license, if any, or (ii) to replace the defective media on which the Software is contained. You must return the defective media to Netop Business Solutions at your expense with a copy of your receipt. This limited warranty is void if the defect has resulted from accident, abuse, or misapplication. Any replacement media will be warranted for the remainder of the original warranty period. This remedy is not available to the extent that Netop Business Solutions is subject to restrictions under export control laws and regulations.

c. Warranty Disclaimer. THE SOFTWARE IS PROVIDED "AS IS" and "AS AVAILABLE". TO THE MAXIMUM EXTENT PERMITTED BY APPLICABLE DANISH LAW, NETOP BUSINESS SOLUTIONS DISCLAIMS ALL WARRANTIES, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NONINFRINGEMENT WITH RESPECT TO THE SOFTWARE AND THE ACCOMPANYING DOCUMENTATION. YOU ASSUME RESPONSIBILITY FOR SELECTING THE SOFTWARE TO ACHIEVE YOUR INTENDED RESULTS, AND FOR THE INSTALLATION OF, USE OF, AND RESULTS OBTAINED FROM THE SOFTWARE. WITHOUT LIMITING THE FOREGOING PROVISIONS, NETOP BUSINESS SOLUTIONS UNDERTAKES NO LIABILITY THAT THE SOFTWARE WILL BE ERROR-FREE OR FREE FROM INTERRUPTIONS, PROBLEMS CO-WORKING WITH OTHER SOFTWARE PACKAGES OR OPERATING SYSTEMS, OR ANY OTHER FAILURES, OR THAT THE SOFTWARE WILL MEET YOUR REQUIREMENTS, WHETHER EXPRESSLY ACKNOWLEDGED OR NOT.

d. Limitation of Liability. UNDER NO CIRCUMSTANCES AND UNDER NO LEGAL THEORY, WHETHER IN TORT, CONTRACT, OR OTHERWISE, SHALL NETOP BUSINESS SOLUTIONS OR ITS SUPPLIERS BE LIABLE TO YOU OR TO ANY OTHER PERSON (PHYSICAL OR LEGAL) FOR ANY INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES OF ANY CHARACTER INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF GOODWILL, WORK STOPPAGE, COMPUTER FAILURE, LOSS OF DATA OR MALFUNCTION, OR FOR ANY AND ALL OTHER DAMAGES OR LOSSES. IN NO EVENT WILL NETOP BUSINESS SOLUTIONS BE LIABLE FOR ANY DAMAGES IN EXCESS OF THE LIST PRICE NETOP BUSINESS SOLUTIONS CHARGES FOR A LICENSE TO THE SOFTWARE, EVEN IF NETOP BUSINESS SOLUTIONS SHALL HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. THIS LIMITATION OF LIABILITY SHALL NOT APPLY TO LIABILITY FOR DEATH OR PERSONAL INJURY TO THE EXTENT ONLY THAT APPLICABLE LAW PROHIBITS SUCH LIMITATION. The foregoing provisions shall be enforceable to the maximum extent permitted by applicable law. 


8. High Risk Activities. The Software is not fault-tolerant and is not designed or intended for use in hazardous environments that require fail-safe performance, including without limitation, in the operation of nuclear facilities, aircraft navigation or communication systems, air traffic control, weapons systems, direct life-support machines, or any other application in which the failure of the Software could lead directly to death, personal injury, or severe physical or property damage collectively, "High Risk Activities". Netop Business Solutions expressly disclaims any express or implied warranty of fitness for High Risk Activities. 

9. Miscellaneous. This Agreement is governed by the laws of Denmark, without reference to conflict of law principles. The application of the United Nations Convention of Contracts for the International Sale of Goods is expressly excluded. This Agreement sets forth all rights for the user of the Software and is the entire agreement between the parties. This Agreement supersedes any other communications with respect to the Software and Documentation. This Agreement may not be modified except by a written addendum issued by Netop Business Solutions. No provision hereof shall be deemed waived unless such waiver shall be in writing and signed by Netop Business Solutions. If any provision of this Agreement is held invalid, the remainder of this Agreement shall continue in full force and effect. The provision, which is held invalid, shall to the furthest extent possible be legally interpreted and construed in accordance with the intentions and the nature of this entire Agreement. The parties confirm that it is their wish that this Agreement has been written in the English language only. Nothing contained in this section 9 shall be construed as an obligation by you to disclose any of your proprietary or confidential information to any third party. 

10. Your representations and warranties. You represent and warrant that you are authorized to enter into this Agreement and comply with its terms. Furthermore, you represent and warrant that you will at any and all times comply with your obligations hereunder and any applicable laws, regulations and policies, which may apply to the Software. You assume full and unlimited liability for any use contrary to this entire Agreement whether such use has been enacted or caused directly or indirectly by you. You agree to indemnify, defend and hold Netop Business Solutions harmless from and against any and all liability arising out of or in connection with your violation or breach of this Agreement or applicable laws, regulations and policies, or Your negligence or willful misconduct 

11. NETOP BUSINESS SOLUTIONS CUSTOMER CONTACTS. If you have any questions concerning these terms and conditions, or if you would like to contact Netop Business Solutions for any other reason, please write to Netop Business Solutions A/S, Bregnerodvej 127, DK-3460 Birkerod, Denmark or visit our website www.netop.com for detailed contact information. 

Netop Business Solutions A/S End-User License Agreement Appendices

Appendix A: Products purchased as perpetual licenses

Use of perpetual licenses: The Software is licensed as a single product; it may not be used on more than one Client Device or by more than one user at a time (unless Your License Certificate particularly authorizes such use). The Software is "in use" on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or installed into the permanent memory (e.g., hard disk, CD-ROM, or other storage device) of that Client Device. This license authorizes You to make one copy of the Software solely for backup or archival purposes, provided that the copy You make contains all of the proprietary notices for the Software.


1. Server Use. To the extent that the applicable order acknowledgement, product invoice, product packaging or License Certificate sets forth, you may use the Software on a Client Device or on a Server within a multi-user or networked environment ("Server Use"). A separate license is required for each Client Device or "seat" that may connect to the Software at any time, regardless of whether such licensed Client Devices or seats are connected to the Software concurrently, or are actually using the Software at any particular time. Your use of software or hardware that reduces the number of Client Devices or seats that connect to and use the Software directly or simultaneously (e.g., "multiplexing" or "pooling" software or hardware) does not reduce the number of licenses required. Specifically, You must have that number of licenses that would equal the number of distinct inputs to the multiplexing or pooling software or hardware "front end"). If the number of Client Devices or seats that can connect to the Software can exceed the number of licenses You have obtained, then You must have a reasonable mechanism in place to ensure that Your use of the Software does not exceed the use limits specified for the license You have obtained. This license authorizes You to make or download one copy of the Documentation for each Client Device or seat that is licensed, provided that each such copy contains all of the proprietary notices for the Documentation.

2. Volume Use. If the Software is licensed with volume use terms specified in the applicable order acknowledgement, product invoice, License Certificate or product packaging, You may make, install and use as many additional copies of the Software on the number of Client Devices as the terms specify. You must have a reasonable mechanism in place to ensure that the number of Client Devices on which the Software has been installed does not exceed the number of licenses you have obtained. This license authorizes You to make or download one copy of the Documentation for each additional copy authorized by the volume license, provided that each such copy contains all of the proprietary notices for the Documentation.

3. Concurrent Use. If the Software is licensed with Concurrent Use terms as specified in the applicable order acknowledgement, product invoice, License Certificate or product packaging, You may install the associated Software on an unlimited number of Client Devices .The Software may be used or accessed  on as many simultaneous sessions according to the number of  purchased accesses according to the applicable order acknowledgement, product invoice, License Certificate or product packaging . The Software is "in use" on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or accessed by You or an employee or authorized agent of You. Should the same employee or authorized agent of You have multiple simultaneous accesses to the Software, each access shall be separately considered a Concurrent User A license for a Concurrent User only grants access rights to individuals; devices and machines may not utilize a Concurrent User license to access to the Software.

4. Vendor signing. For the Android version of Netop Mobile and Embedded it is explicitly prohibited for end-users or any other to vendor sign or cause vendor signing of this module and thereby enabling this module to reach necessary system resources needed for the viewing of the screen and injection of input to the devices, that it otherwise could not reach. Doing so immediately revokes the license to use this module.

5. Services and features depending on Netop Advantage Annual Service & Upgrade Program.  Vision Pro Learning Center is a feature running as a service (SaaS). The feature is included with Vision Pro for as long as You have an active Netop Advantage program and continuous use of Learning Center Service is dependent on Your subscription to the Netop Advantage Program. If Your Netop Advantage subscription expires You will no longer be able to access and use Vision Pro Learning Center. Use of the Learning Center feature is subject and regulated by this agreement appendix C.  Netop will make storage space available in the Vision Pro Learning Center. Any storage space made available by Netop will have limitations on such storage space as set by Netop from time to time, and the available storage space can be subject to change by Netop without your consent given no less than 30 days notice.  

6. Consent to Use of Data. You agree that Netop Business Solutions may collect and use anonymous technical data and related information, including but not limited to technical information about Your device, system and application software, and peripherals, that is gathered periodically to facilitate the provision of software updates, product support and other services to You (if any) related to the Licensed Software. Netop Business Solutions may use this information, as long as it is in a form that does not personally identify You, to improve its products or to provide services or technologies to You.


Netop Advantage Annual Service & Upgrade Program: Your software license includes 12 months of free product service, support and software upgrade(s). After 12 months a renewal of Netop Advantage is available for a fee of 20% of the current list price for your software license. A renewal allows you access to Netop support services, resolution of support cases according to Netop’s Service Level Agreement and full investment protection in terms of free software upgrades when available. When Your Netop Advantage subscription expires the above mentioned services will no longer be available. Please refer to www.netop.com/support-overview further details about support services and the Netop Advantage program.

Appendix B: Products purchased as subscriptions

THIS SUBSCRIPTION AGREEMENT (“AGREEMENT”) GOVERNS YOUR FREE TRIAL OF THE PRODUCTS. IF YOU PURCHASE OUR PRODUCTS, THIS AGREEMENT WILL ALSO GOVERN YOUR PURCHASE AND ONGOING USE OF THOSE PRODUCTS. 

BY ACCEPTING USE OF FREE TRIAL OR BY EXECUTING AN ORDER FORM THAT REFERENCES THIS AGREEMENT, YOU AGREE TO THE TERMS OF THIS AGREEMENT. IF YOU ARE ENTERING INTO THIS AGREEMENT ON BEHALF OF A COMPANY OR OTHER LEGAL ENTITY, YOU REPRESENT THAT YOU HAVE THE AUTHORITY TO BIND SUCH ENTITY AND ITS AFFILIATES TO THESE TERMS AND CONDITIONS, IN WHICH CASE THE TERMS "YOU" OR "YOUR" SHALL REFER TO SUCH ENTITY AND ITS AFFILIATES. IF YOU DO NOT HAVE SUCH AUTHORITY, OR IF YOU DO NOT AGREE WITH THESE TERMS AND CONDITIONS, YOU MUST NOT ACCEPT THIS AGREEMENT AND MAY NOT USE THE SERVICES. 

1. DEFINITIONS

"Affiliate" means any entity which directly or indirectly controls, is controlled by, or is under common control with the subject entity. "Control," for purposes of this definition, means direct or indirect ownership or control of more than 50% of the voting interests of the subject entity.

“Documentation” means User Guide, quick guide, FAQ, knowledge base articles, release notes or other documentation specific to the Products.

"Malicious Code" means viruses, worms, time bombs, Trojan horses and other harmful or malicious code, files, scripts, agents or programs.

"Order Form" means the ordering documents for purchases hereunder, including addenda thereto, that are entered into between You and Netop Business Solutions from time to time. 

“Products” means on-line or offline software products provided by Netop Business Solutions at www.netop.com or by Netop Business Solutions’ Affiliates at their websites or stores as subscriptions.

"Users" means individuals who are authorized by You to use the Products, for whom subscriptions to a Product have been purchased, and who have been supplied with a specific and unique identification and password by You or by Netop Business Solutions identifying the specific user. Users may include but are not limited to Your employees, consultants, contractors and agents; or third parties with which You transact business.

2. USE OF FREE TRIAL

(a) Use of free trial: Netop Business Solutions makes Products available to You on a trial basis free of charge until the (a) the fourteenth (14) or the thirtieth (30) day after Your acceptance of this Agreement or (b) the start date of any purchased Products ordered by You. You are informed about additional trial terms and conditions at the trial registration web page. The free trial period may vary depending the type or version of the Products. You will be informed about the length of the trial at download. Any such additional terms and conditions are incorporated into this Agreement by reference and are legally binding. 

ANY DATA YOU ENTER INTO THE PRODUCTS, AND ANY CUSTOMIZATIONS MADE TO THE PRODUCTS BY OR FOR YOU, DURING YOUR FREE TRIAL MAY BE LOST UNLESS YOU PURCHASE A SUBSCRIPTION TO THE SAME PRODUCTS AS THOSE COVERED BY THE TRIAL, PURCHASE UPGRADED PRODUCTS, OR EXPORT SUCH DATA, BEFORE THE END OF THE FREE TRIAL PERIOD. YOU CANNOT TRANSFER DATA ENTERED OR CUSTOMIZATIONS MADE DURING THE FREE TRIAL TO A PRODUCT THAT WOULD BE A DOWNGRADE FROM THAT COVERED BY THE TRIAL (E.G. BASIC TO FREE VERSION) THEREFORE, IF YOU PURCHASE A SERVICE THAT WOULD BE A DOWNGRADE FROM THAT COVERED BY THE TRIAL, YOU MUST EXPORT YOUR DATA BEFORE THE END OF THE FREE TRIAL PERIOD OR YOUR DATA RISK BEING PERMANENTLY LOST. 

NOTWITHSTANDING SECTION 7 (WARRANTY AND DISCLAIMERS), DURING THE FREE TRIAL THE SERVICES ARE PROVIDED “AS-IS” AND “AS AVAILABLE” WITHOUT ANY WARRANTY. 

(b) Use of special versions: Some Products will at the end of a free trial period where no purchase was made or at the end of a subscription term where no renewal was purchased turn into a free version with limited functionality. You will be informed about free versions of the Products at download. Free versions can be used free of charge. Terms and conditions for use of free versions of the Products are incorporated into this Agreement by reference and are legally binding. 

3. USE OF THE PRODUCTS

3.1. Netop Business Solutions’ Responsibility. Netop Business Solutions or at Netop Business Solutions sole choice, Distributors and Partners appointed by Netop Business Solutions, shall provide  You basic support for the purchased Products at no additional charge according to Netop Business Solutions standard Service Level Agreement which can be found at www.netop.com. Netop Business Solutions will notify You of updates or new versions of the Products and any download of an updated version or new versions of the Products confirms your acceptance of this version. Information of new product updates and content will be available at www.netop.com and in the Documentation to the applicable product. Unless otherwise specified in the Order Form or Documentation, updates are included in the purchased Products.

3.2. Your Responsibilities. You shall (a) be responsible for Users’ compliance with this Agreement, (b) be solely responsible for the accuracy, quality, integrity and legality of Your Data and of the means by which You acquired Your Data, (c) use commercially reasonable efforts to prevent unauthorized access to or use of the Products, and notify Netop Business Solutions promptly of any such unauthorized access or use, and (iv) use the Products only in accordance with the User Guide or applicable Documentation and in accordance with any applicable law and/or government regulations. You shall not (a) make the Products available to anyone other than Users, (b) sell, resell, rent or lease the Products, (c) use the Products to store or transmit infringing, libelous, or otherwise unlawful or tortious material, or to store or transmit material in violation of third-party privacy rights, (d) use the Products to store or transmit Malicious Code, (e) interfere with or disrupt the integrity or performance of the Products or third-party data contained therein, (f) attempt to gain unauthorized access to the Products or their related systems or networks, or (g) use the Product to monitor, supervise, spy or log personal data if such usage is in violation with privacy rights or local laws. You or Your Users shall be responsible for complying with all domestic and international laws and any other any legal obligations, including but not limited to, obligations imposed by copyright, trademark, trade secrets, defamation, decency, privacy, security and export laws. 

3.3. Usage Limitations. Products may be subject to other limitations, such as, for example, limits on disk storage space, bandwidth, platform support or similar. Any such limitations are specified in the product Documentation. 

3.4. Rights to data. All data originating from use of the products saved in a memory, hard disk or any other subject are wholly owned by You as user or Customer to Netop Business Solutions. Netop Business Solutions is not liable for any content of such information. 

3.5. Storage of data. Netop Business Solutions will storage all information as one of the following:

1. Data will be stored for the duration of this agreement governing the purchase of a service on Netop Business Solutions servers.
2. Data will be stored for a fixed period as stated in the Documentation of the service. That means that data that exceeds that fixed period will be automatically removed. This is done in a chronological order – older data is removed and more recent is maintained.

At the termination date Netop Business Solutions will store all data (or all remaining data) for thirty (30) days after which the Data automatically will be deleted on Netop Business Solutions server. You are solely responsible for the retrieval and storage of all Data owned by You as part of Your use of the Service provided by Netop Business Solutions, and under no circumstance can Netop Business Solutions be held responsible for deleted data owned by You after the 30 days has expired from the date of termination.  

4. PURCHASE AND PAYMENT TERMS

4.1. Provision of Purchased Products. Netop Business Solutions shall make the purchased Products available to You pursuant to this Agreement and the relevant Order Forms during the subscription term. You agree that Your purchases hereunder are neither contingent on the delivery of any future functionality or features nor dependent on any oral or written public comments made by Netop Business Solutions regarding future functionality or features.

4.2. User Subscriptions. Unless otherwise specified in the applicable Order Form, Products are purchased as User subscriptions and may be accessed by no more than the specified number of Users. The user identification and password can only be used by one specific user and cannot be transferred or used by other users. Additional User subscriptions may be added at any time. User subscriptions are for designated Users and cannot be shared or used by more than one User at a time but may be reassigned to new Users replacing former Users who no longer require ongoing use of the Products.

4.3. Concurrent Subscriptions. The Concurrent Subscription gives You the right to install the associated Software on an unlimited number of Client Devices. The Software may be used on as many simultaneous sessions according to the number of subscriptions purchased. The Software is "in use" on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or installed into the permanent memory (e.g., hard disk, CD-ROM, or other storage device) of that Client Device.

4.4. User Fees. You shall pay all fees specified in all Order Forms hereunder. Except as otherwise specified herein or in an Order Form, (a) fees are based on Products purchased or actual usage of Products, (b) payment obligations are non-cancelable and fees paid are non-refundable, and (c) the number of User subscriptions purchased cannot be decreased during the relevant subscription term stated on the Order Form. User subscription fees are based on monthly or annual payments that begin on the subscription start date and each monthly anniversary thereof or each twelve (12) months.

4.5. Invoicing and Payment. You will provide Netop Business Solutions with valid and updated credit card information or with a valid purchase order or alternative document reasonably acceptable to Netop Business Solutions.  If You provide credit card information to Netop Business Solutions, You authorize Netop Business Solutions to charge such credit for all Products and/or Services listed in the Order Form for the initial subscription term and any renewal subscription term(s).  Such charges shall be made in advance, either annually or in accordance with any different billing frequency stated in the applicable Order Form. If the Order Form specifies that payment will be by a method other than a credit card, Netop Business Solutions will invoice You in advance and otherwise in accordance with the relevant Order Form. Unless otherwise stated in the Order Form, invoiced charges are due net 30 days from the invoice date. You are responsible for maintaining complete and accurate billing and contact information to Netop Business Solutions.

4.6. Overdue Charges. If any charges are not received from You by the due date, then at Netop Business Solutions discretion, (a) such charges may accrue late interest at the rate of 1.5% of the outstanding balance per month, or the maximum rate permitted by law, whichever is lower, from the date such payment was due until the date paid, and/or (b) Netop Business Solutions may condition future subscription renewals and Order Forms on payment terms shorter than those specified in Section 4.5 (Invoicing and Payment).

4.7. Suspension of Service and Acceleration. If any amount owing by You under this or any other agreement for Netop Business Solutions services is 30 or more days overdue, Netop Business Solutions may, without limiting other rights and remedies, accelerate Your unpaid fee obligations under such agreements so that all such obligations become immediately due and payable, and suspend Your Product usage until such amounts are paid in full.

4.8. Taxes. Unless otherwise stated, Netop Business Solutions’ fees do not include any taxes, levies, duties or similar governmental assessments of any nature, including but not limited to value-added, sales, use or withholding taxes, assessable by any local, state, provincial, federal or foreign jurisdiction (collectively, " Taxes"). You are responsible for paying all Taxes associated with Your purchases hereunder. If Netop Business Solutions has the legal obligation to pay or collect Taxes for which You are responsible under this paragraph, the appropriate amount shall be invoiced to and paid by You, unless You provide Netop Business Solutions with a valid tax exemption certificate authorized by the appropriate taxing authority. For clarity, Netop Business Solutions is solely responsible for taxes assessable against it based on Netop Business Solutions’ income, property and employees.

5. Consent to Use of Data. You agree that Netop Business Solutions may collect and use anonymous technical data and related information, including but not limited to technical information about Your device, system and application software, and peripherals, that is gathered periodically to facilitate the provision of software updates, product support and other services to You (if any) related to the Licensed Software. Software Provider Netop Business Solutions may use this information, as long as it is in a form that does not personally identify You, to improve its products or to provide services or technologies to You.

6. Service & Upgrade: Products purchased as subscriptions includes free product service, support and software upgrade(s). Please refer to www.netop.com/support-overview for further details about Netop’s support services.

Appendix C: Products provided as hosted services

1. Use of hosted Services. Netop Business Solutions provides hosted solutions and the software enabling the use thereof and related updates, including any documentation files accompanying the software (the software, any modifications or updates and documentation hereinafter collectively referred to as the "Software"). Software and Service are provided by Netop Business Solutions and/or its affiliates and assigned to You either as a Service hosted by Netop Business Solutions or by You. From time to time Netop Business Solutions will notify You of updates or new versions of the Software or Service. Your download of the updated or new versions of the Software and/or subsequent use of the Service confirms your acceptance of, and is also subject to, these Terms and Conditions. If you do not agree to abide by these Terms and Conditions, you may not use the Service or Software

2. General conditions and restrictions. Your responsibilities in connection with Services hosted by Netop Business Solutions: Netop Business Solutions strives to protect personal information and privacy, however, Netop Business Solutions cannot guarantee the security of any information You disclose online. You acknowledge by entering into this Agreement that you are aware of our limitations with respect to security and privacy. You agree to provide true, accurate, current and complete information as required by Netop Business Solutions. Unless produced by Netop Business Solutions, the information You access or receive by using the Software, and/or the Service, and information sent to you by visitors to your website, is provided by the visitors and is not, controlled, verified or endorsed by Netop Business Solutions in any way. 

By using the Software and the Service you agree to: (a) determine whether the information complies with Your needs; (b) determine whether You have adequate legal rights to store, reproduce or otherwise make use of the information in the manner contemplated by you; and (c) comply with all domestic and international laws and any other any legal obligations, including but not limited to, obligations imposed by copyright, trademark, trade secrets, defamation, decency, privacy, security and export laws. 
You understand and acknowledge that Netop Business Solutions has no obligation to monitor any portion of the Service, including without limitation, content posted by You or Your visitors. You acknowledge and agree, however, that Netop Business Solutions retains the right to monitor the Service, and to disclose any information as necessary or appropriate to satisfy any law, regulation or other governmental request, to operate the Service properly, or to protect the Service. Netop Business Solutions reserves the right to refuse to transmit or to remove any information or materials, in whole or in part, that, in Netop Business Solutions’ sole discretion, Netop Business Solutions deems unacceptable, undesirable, inappropriate or in violation of any law or these Terms and Conditions. You agree to immediately notify us of any unauthorized use of the Service or the Software or any other breach of security known to or suspected by you. Netop Business Solutions assumes no responsibility for the deletion of or failure to store electronic messages, chat communications, or other information submitted by You or Your visitors using the Software or Service, whether submitted as part of the Service or otherwise. 

If You violate these Terms and Conditions, You may be permanently or temporarily banned from using the Service and Software, at Netop Business Solutions' sole discretion. By way of example but not limitation, Netop Business Solutions reserves the right to terminate this Agreement at any time if Netop Business Solutions determines, in its sole discretion, that Your site or activities are unsuitable for the Service. "Unsuitable" sites or activities may include, without limitation, those sites that Netop Business Solutions believes: (a) display or promote obscene, indecent, pornographic or sexually explicit activities; (b) promote abusive, threatening, hateful, harmful, harassing, libelous or violent behavior; (c) promote discrimination based on race, sex, religion, nationality, disability, sexual orientation, or age; (d) promote illegal activities or fail to comply with applicable law; (e) violate the intellectual property rights of third parties; or (f) have the potential to interfere with the use, enjoyment, deployment or operation of the Service by other visitors.

3. Rights to data. All data originating from use of the products saved in a memory, hard disk or any other subject are wholly owned by You as user or Customer to Netop Business Solutions. Netop Business Solutions is not liable for any content of such information. 

4. Storage data. Netop Business Solutions will store all information as one of the following:

1. Data will be stored for the duration of this agreement regulating the purchase of a service on Netop Business Solutions servers.
2. Data will be stored for a fixed period as stated in the documentation of the service. That means that data that exceeds that fixed period will be automatically removed. This is done in a chronological order – older data is removed and more recent is maintained.

At the termination date Netop Business Solutions will storage all data for 30 days after which the Data automatically will be deleted on Netop Business Solutions server. You are sole responsible for the retrieval and storage of all Data owned by You as part of your use of the Service provided by Netop Business Solutions, and under no circumstance can Netop Business Solutions be held responsible for deleted data owned by You after the 30 days has expired from the date of termination.

5. Consent to Use of Data. You agree that Netop Business Solutions may collect and use anonymous technical data and related information, including but not limited to technical information about Your device, system and application software, and peripherals, that is gathered periodically to facilitate the provision of software updates, product support and other services to You (if any) related to the Licensed Software. Netop Business Solutions may use this information, as long as it is in a form that does not personally identify You, to improve its products or to provide services or technologies to You.
  
6. Service & Upgrade: Products purchased as subscriptions include free product service, support and software upgrade(s). Please refer to www.netop.com/support-overview for further details about Netop’s support services.

Appendix D: Products purchased from App Stores

1. Acknowledgement. Netop Business Solutions and You acknowledge that this Agreement is between Netop Business Solutions and You only and not with Apple, Inc. (“Apple”) or Google Inc. (“Google”). Netop Business Solutions, not Apple or Google, is solely responsible for the Software and the content thereof. 

2. Scope of License. The license granted to the Software is limited to a non-transferable license to use the Software on any iPad, iPod Touch, iPhone and/or Android tablet that You own or control and as permitted by the Usage Rules set forth in the App Store Terms of Service – http://www.apple.com/legal/itunes/us/terms or in the Google Play Terms of Service – http://play.google.com/intl/en/about/play-terms.html

3. Maintenance and Support. Netop Business Solutions is solely responsible for providing any maintenance and support services with respect to the Software, as specified in the Agreement, or as required under applicable law. Netop Business Solutions and You acknowledge that Apple or Google has no obligation whatsoever to furnish any maintenance and support services with respect to the Software.

4. Warranty. Netop Business Solutions is solely responsible for any product warranties, whether express or implied by law, to the extent not effectively disclaimed. In the event of failure of the Software to conform to any applicable warranty, You may notify Apple or Google, and Apple or Google will refund the purchase price for the Software to You; and that, to the maximum extent permitted by applicable law, Apple or Google will have no other warranty obligation whatsoever with respect to the Software, and any other claims, losses, liabilities, damages, costs or expenses attributable to any failure to conform to any warranty will be Netop Business Solutions’ sole responsibility.

5. Product Claims. Netop Business Solutions and You acknowledge that Netop Business Solutions, not Apple or Google, is responsible for addressing any claims of You or any third party relating to the Software and Your possession and/or use of that Software, including, but not limited to: (i) product liability claims; (ii) any claim that the Software fails to conform to any applicable legal or regulatory requirement; and (iii) claims arising under consumer protection or similar legislation. 

6. Intellectual Property Rights. Netop Business Solutions and You acknowledge that, in the event of any third party claim that the Software or Your possession and use of that Software infringes that third party’s intellectual property rights, Netop Business Solutions, not Apple or Google, will be solely responsible for the investigation, defense, settlement and discharge of any such intellectual property infringement claim.

7. Legal Compliance. You must represent and warrant that (i) he/she is not located in a country that is subject to a U.S. Government embargo, or that has been designated by the U.S. Government as a “terrorist supporting” country; and (ii) he/she is not listed on any U.S. Government list of prohibited or restricted parties.

8. Third Party Beneficiary. Netop Business Solutions and You must acknowledge and agree that Apple,  Apple’s subsidiaries, Google and Google’s subsidiaries, are third party beneficiaries of the Agreement, and that, upon Your acceptance of the terms and conditions of the Agreement, Apple or Google will have the right (and will be deemed to have accepted the right) to enforce the Agreement against You as a third party beneficiary thereof.

9. Consent to Use of Data. You agree that Netop Business Solutions may collect and use anonymous technical data and related information, including but not limited to technical information about Your device, system and application software, and peripherals, that is gathered periodically to facilitate the provision of software updates, product support and other services to You (if any) related to the Licensed Software. Netop Business Solutions may use this information, as long as it is in a form that does not personally identify You, to improve its products or to provide services or technologies to You.


10. Contact Information. Should you have any questions, complaints or claims with respect to the Agreement, please contact us at Netop Business Solutions A/S, Bregnerodvej 127, DK-3460 Birkerod, Denmark or +45 4590 25 25 or info@netop.com. 

11. Vendor signing. For the Android version of Netop Mobile and Embedded it is explicitly prohibited for end-users or any other to vendor sign or cause vendor signing of this module and thereby enabling this module to reach necessary system resources needed for the viewing of the screen and injection of input to the devices, that it otherwise could not reach. Doing so immediately revokes the license to use this module.



