#!/usr/bin/perl -w
#
# configure Netop Host installed using rpm or tar.gz
# (c) Netop Business Solutions A/S 2016
#

$VERSION = "configure: 1.07";
$DEBUG = 0;
$NETOPHOST = "/usr/sbin/netophost";
$QXMLEDIT = "/usr/bin/qxmledit";
$NETOPHOSTXD = "/usr/sbin/netophostxd";
$INITD_NETOPHOSTD = "/etc/init.d/netophostd";

$UNCONFIGURE_SCRIPT = "/etc/NetopHost/cfg/unconfigure.pl";
$UNCONFIGURE_COMMANDS = "";
#$LICFILE = "/etc/NetopHost/LICENSE";
$INIFILE = "/etc/NetopHost/host.xml";

use Cwd;
use File::Basename;

my $CHKCONFIG_CMD = "/sbin/chkconfig";
my $CP_CMD    = "/bin/cp";
my $MV_CMD    = "/bin/mv";
my $LN_CMD    = "/bin/ln";
my $LS_CMD    = "/bin/ls";
my $FIND_CMD  = "/usr/bin/find";
my $CAT_CMD   = "/bin/cat";
my $ID_CMD    = "/usr/bin/id";
my $D2U_CMD   = "/usr/bin/dos2unix";
my $CHMOD_CMD = "/usr/bin/chmod";

my $SKIP=`awk '/^__LICENSE_FOLLOWS__/ { printf("%d", NR + 1); exit 0; }' $0`;
my $CRTDIR=cwd();
my $THIS="$0";

my $USER_LICENSE_FILE = "license.dat";
my $LICENSE_DESTINATION = "/etc/NetopHost/cfg";
my $LOCAL_LICENSE_FILE = ".license.dat";

my $SYSTEM = "";
my $HOSTXD_WORKS = "no";
my $OS_VER = "";
my $OS_BIT = "";
my $QUERY_RES = "start_status:";

sub write_unconfigure
{
   open OUTFILE,">$UNCONFIGURE_SCRIPT" or die "Cannot open '$UNCONFIGURE_SCRIPT' for writing: $!\n";
   printf OUTFILE "%s",$UNCONFIGURE_COMMANDS;
   close OUTFILE;
}

sub system_setup
{
    my $uname = `uname`;
    dbg("uname  = $uname");
    if ($uname =~ "SunOS") {
          $SYSTEM = "solaris";
          $OS_VER = `uname -r`;
    } elsif ($uname =~ "Linux") {
          $SYSTEM = "Linux";
          my $release = "/etc/redhat-release";
          if (-e $release) {
                $SYSTEM = "RedHat";
                open FILE,"<$release";
                $line=<FILE>;
                if($line eq "Red Hat Enterprise Linux AS release 4 (Nahant)") {
                    $HOSTXD_WORKS="yes";
                }
          }
          $release = "/etc/SuSE-release";
          if (-e $release) {
                $SYSTEM = "SuSE";
          }
          $release = "/etc/depmod.d/ubuntu.conf";
          if (-e $release) {
                $SYSTEM = "ubuntu";
          }
    } else {
          dbg("Warning: cannot identify UNIX system.\n");
          return 0;
    }
    $OS_VER =~ s/\n//g;
	$OS_BIT = `uname -m`;
    dbg("SYSTEM = $SYSTEM\n");
    dbg("OS_VER = $OS_VER\n");
    return 1;
}

sub get_tool
{
    my ($arg) = @_;
    my $tool="";
    my $cmd = "unalias cp 2>/dev/null; which $arg 2>/dev/null";
    dbg("cmd: $cmd\n");
    $tool = `$cmd`;
    $tool =~ s/\n//g;
    if($tool =~ m|[ \t]|) {
       $tool = ""; # contained whitespace, must be a message.
    }

    return $tool;
}

sub setup_tools
{
    if(!(-e $CHKCONFIG_CMD)) {
        $CHKCONFIG_CMD =  get_tool("chkconfig");
    }
    dbg("setup_tools: CHKCONFIG_CMD = $CHKCONFIG_CMD\n");
    if(! -e $CP_CMD) {
        $CP_CMD = get_tool("cp");
        if(! -e $CP_CMD ) {
            die "Non-standard cp, please update CP_CMD\n";
        }
    }
    if(! -e $LN_CMD) {
        $LN_CMD = get_tool("ln");
        if(! -e $LN_CMD) {
            die "Non-standard ln, please update LN_CMD)\n";
        }
    }
    if(!-e $LS_CMD) {
        $LS_CMD = get_tool("ls");
        if(! -e $LS_CMD ) {
            die "Non-standard ls, please update LS_CMD\n";
        }
    }
    if(! -e $FIND_CMD) {
        $FIND_CMD = get_tool("find");
    }
    if(! -e $CAT_CMD) {
        $CAT_CMD = get_tool("cat");
        if(! -e $CAT_CMD) {
            die "Non-standard ln, please update CAT_CMD)\n";
        }
    }
    if(! -e $ID_CMD) {
        $ID_CMD = get_tool("id");
    }
    if(! -e $D2U_CMD) {
        $D2U_CMD = get_tool("dos2unix");
        if(! -e $D2U_CMD ) {
            $D2U_CMD = get_tool("fromdos");
            if(! -e $D2U_CMD) {
                die "Non-standard dos2unix, please update D2U_CMD\n";
            }
        }
    }
    if(! -e $CHMOD_CMD) {
        $CHMOD_CMD = get_tool("chmod");
        if(! -e $CHMOD_CMD ) {
            die "Non-standard chmod, please update CHMOD_CMD\n";
        }
    }
}

sub dbg
{
    if($DEBUG) {
    my $arg = shift(@_);
    print STDERR "DBG: $arg";
    }
}

sub input_confirmation
{
   my $options = "[yes/No/quit]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("y"))) {
           return "y";
       } elsif (($_ eq "\n") || ($lower =~ ("n"))) {
           return "n";
       } elsif (($lower =~ ("q"))) {
           return "q";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub input_confirmation_Y
{
   my $options = "[Yes/no/quit]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("n"))) {
           return "n";
       } elsif (($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } elsif (($lower =~ ("q"))) {
           return "q";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

#my $TRIAL_LICENSE=

#my $NML_LICENSE=

sub write_licfile
{
   my ($text,$licfile) = @_;
   open OUTFILE,">$licfile" or die "Cannot open '$licfile' for writing: $!\n";
   printf OUTFILE "%s",$text;
   close OUTFILE;
   print "License agreement written to $licfile\n\n";
}

# $_[0] is current license file (MANDATORY)
# $_[1] is destination folder   (MANDATORY)
# $_[2] in the new name of the license file to copy into destination (OPTIONAL) == basename($_[0])
# //
sub copy_license_to_installation_point
{
    ( $source, $destination, $new_name ) = @_;

    my $error = "\tInvalid arguments in : sub copy_license_to_installation_point.\n";
    goto return_error unless ( defined $source ) and ( defined $destination ); # sanity check
    $error = "\tInvalid license file : '$source' .\n";
    goto return_error unless ( (! -e $source ) or ( ! -f $source ) );
    
    $new_name = basename( $source ) unless defined $new_name;
    my $new_license = safe_license_to_target( $new_name, $destination );
    # copy the current license
    return $new_license unless ( 0 != system("$CP_CMD $source $destination_file") );
    $error = "\tFailed to copy '$source' to '$destination_file'!\n";
return_error:
    print STDERR "$error";
    return undef;
} # sub copy_license_to_installation_point

# generate the path target for new license file
# if there already is such a file, rename it to something like <old_file>.<[0-9]+>
# $_[0] is license file name
# $_[1] is target path
sub safe_license_to_target
{
    ( $license_file, $destination ) = @_;

    my $error = "\nInvalid arguments in : sub safe_license_to_target.\n";
    goto return_error unless ( defined $license_file ) and ( defined $destination ); # sanity check

    $error = "\tDestination '$destination' is invalid ( not a directory ) !\n";
    goto return_error unless ( -d $destination ); # destination is a directory ?

    $error = "\tDestination '$destination' is invalid ( not writable ) !\n";
    goto return_error unless ( -w $destination ); # destination is writable ?

    my $destination_file = $destination . '/' . $license_file;
    if ( -e $destination_file ) # already exists ?
    {
        $error =  "\tLocal license copy '$destination_file' already exists and it not a regular file !\n";
        goto return_error unless ! ( -f $destination_file );

        # make a backup copy with a random suffix :
        my $prefix_num = 0;
        my $previous_license_file = $destination_file;
        do
        {
            $previous_license_file = $destination_file . '.' .  $prefix_num;
        }
        while ( ! -e $previous_license_file );

        print "\tAn older version of '$destination_file' already exists, will save it to $previous_license_file.\n";
        return $destination_file unless ( 0 != system("$MV_CMD $destination_file $previous_license_file") );

        $error = "\tFailed to copy '$destination_file' to '$previous_license_file'!\n";
    } # if ( -e $destination_file )
    return $destination_file;
return_error:
    print STDERR "$error";
    return undef;
} # sub license_to_target

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license
{
    my $scroll_method = "more";
    my $allow_scroll = shift;
    if ( ! defined $allow_scroll ) { $allow_scroll = 1; }
    else { $allow_scroll = 0 + $allow_scroll; } # force numeric ctx
    if ( ! defined $allow_scroll ) { $allow_scroll = 1; }
    my $command = "tail -n +$SKIP $THIS";
    if ( 1 == $allow_scroll ) { $command = $command . " | $scroll_method"; }
    system($command);
} # sub print_license  

sub host_validator_wrapper
{
    ( $serial, $license_path ) = @_;
    
    # (!!) binary that does the actual validation and license generation reads serial from ENVVAR (!!) wtf
    $ENV{"DW_INSTALL_SERIAL"} = $serial;
    return validate_host_serial_number( $license_path );
} # sub host_validator_wrapper

# $_[0] : a license file that user already has ( older license )
# $_[1] : name of the new license file
# $_[2] : destination of new license file
# $_[3] : serial number
# $_[4] : autoinstall (OPTIONAL) = 0 (no)
sub validate_with_license_or_serial
{
    ( $user_license, $new_license, $destination , $serial, $autoinstall ) = @_;

    $autoinstall = 0 unless defined $autoinstall;

    my $new_license_file = undef;
    # user has a serial number :
user_has_serial :
    if ( ( defined $serial ) and ( $serial ne "" ) )
    {
        print "\tWill validate Netop Host with provided license key.\n";
        # determine if there is a license file already ( from a previous install ) and make a copy :
        $new_license_file = safe_license_to_target( $new_license, $destination );
        return ( 0, $new_license_file ) unless defined $new_license_file;
        goto ask_for_serial unless host_validator_wrapper( $serial, $new_license_file );
        return ( 1, $new_license_file );
    }
    # user does not have a serial number :
    # but has a license file :
    elsif ( ( defined $user_license ) and ( -e $user_license ) )
    {
        print "\tFound previous license file : '$user_license' .\n";
        print "\t\tNetop Host will be validated with this licence.\n";
        # copy user's current license to installation directory :
        $new_license_file = copy_license_to_installation_point($user_license, $destination, $new_license);
        return ( 0, $new_license_file ) unless defined $new_license_file;
        return ( 1, $new_license_file ) unless host_validator_wrapper( $serial, $new_license_file );
        print "\n Your license '$user_license' could not be validated.\n";
    }
ask_for_serial:
    # doesn't have a license file
    # interactive install :
    if ( ! $autoinstall )
    {
        $new_license_file = safe_license_to_target( $new_license, $destination ) unless defined $new_license_file;
        return ( 0, $new_license_file ) unless defined $new_license_file;
        ( $validated, $serial ) = patiently_ask_for_serial("Netop Host", \&host_validator_wrapper, 3, $new_license_file);
        return ( 1, $new_license_file ) unless ! $validated;
        return ( 0, $new_license_file );
    }
    return ( 1 , $new_license_file );
} # sub validate_with_license_or_serial

sub host_installed_but_not_validated
{
    print "\nThe application has been installed, but none of the license keys could be validated.\n";
    print "Please run '$THIS' as super-user to configure and validate the product.\n";
    print "Thank you for using Netop Solutions!\n\n";
} # sub host_installed_but_not_validated

sub host_installed_and_validated
{
    my $need_rc = shift;
    $need_rc = 0 unless ( defined $need_rc );
    print "\nNetop Host application has been installed and activated.\n";
    print "Note: display manager needs to be restarted (or machine rebooted) for the changes to take full effect.\n" unless !$need_rc;
    print "Thank you for using Netop Solutions!\n\n";
} # sub host_installed_and_validated

# $_[0] is license_file
sub validate_host_serial_number
{
    my $license_file = shift;
    return 0 unless ( defined $license_file );
    # (!) actual serial number is passed as an environment variable (!)
    print "\tValidating license ...\n";
    my $result = 0;
    if (-e $license_file) {
        $result = `$NETOPHOST --testlic --license.dat $license_file >/dev/null 2>/dev/null; echo \$?`;
    } else {
        $result = `$NETOPHOST --testlic  >/dev/null 2>/dev/null; echo \$?`;
    }
    
    $result =~ s/\s+$//;

    if ( 0 == $result )
    {
        unlink "$license_file"; # was in older implementation
        print "\t\tLicense key validated OK.\n";
        print "\t\tYour license has been saved.\n";
        
        return 1;
    }
    else
    {
        # from older implementation :
        if (($result > 27) && ($result != 101) && ($result != 102))
        {
            print STDERR "Unexpected error from validator $NETOPHOST : $result\nValidation failed!\nPlease contact technical support.\n\n";
            exit(1);
        }
        print "\t\tLicense Key is invalid.\n";
        return 0;
    }
} # sub validate_host_serial_number

# $_[0] is product name to display (MANDATORY)
# $_[1] is the validator subroutine passed as a reference (MANDATORY) that must return a boolean flag.
# $_[2] is the patience ( times to ask ) (OPTIONAL) = 3
# $_[3] is an optional argument to the validator (OPTIONAL) = undef   
sub patiently_ask_for_serial
{
    ( $product_name, $validator, $patience, $validator_arg ) = @_;
    $answer = undef;
    return ( 0, $answer ) unless ( ( defined $product_name ) and ( defined $validator ) );
    
    $patience = 3 unless ( ( defined $patience ) and ( ( $patience *= 1 ) ) ); # default patience : perl is patient

    while ( $patience )
    {
        print "\n\t[$patience] Enter License key for Netop Host or leave blank for trial (press 's' to skip):\n";
        $answer = <STDIN>;
        if ( ( !$answer ) ) { return ( 0, undef ); }
        
        #if enter or skip is pressed, then use the trial license
        if ( ('s' eq lc substr($answer,0,1)) or ($answer =~ /^\s*$/) ) { $answer = "" };        
        
        if ( $validator->( $answer, $validator_arg ) ) { return ( 1, $answer ); }
        $patience--;
        if ($patience) {        
            print "\nThe license key could not be validated, please try again. ($patience)\n";
        }
    }
    
    return ( 0, $answer );
} # sub patiently_ask_for_serial

# $_[0] is the name of the input field to ask for (MANDATORY)
sub ask_for_password_input
{
    system('stty', '-echo');  # Disable echoing
    my $answer = ask_for_input(@_);
    system('stty', 'echo');   # Turn it back on
    print "\n";
    
    return $answer;
}

# $_[0] is the name of the input field to ask for (MANDATORY)
sub ask_for_input
{
    my $patience = 3;
    my $field_name = shift;            
    my $answer = undef;
    
    return undef unless defined ($field_name);
    
    while ( !defined($answer) && $patience--)
    {        
        print "\n$field_name : ";
        chomp($answer = <STDIN>);        
        
        #valid input and not empty and not all spaces
        if ( $answer && $answer ne "" && $answer !~ /^\s*$/) { 
            return $answer; 
        }      
        else { 
            $answer = undef;
        }
    }
    
    return undef;
} # sub ask_for_input

$SIGNAL_UNINSTALL=0;
sub check_license
{
    my ( $serial, $licensedto, $autoinstall ) = @_;
    print "\nThe following license-agreement apply to Netop Host:\n";
    print "[Please read carefully]\n";
    # interactive install
    my $answer = '';
    if ( 0 == $autoinstall )
    {
        print_license( ! $autoinstall );
        $answer = input_confirmation("Do you accept this license ?");
        $answer = lc substr($answer,0,1);
        if ( 'y' ne $answer )
        {
            print "\tLicense not agreed upon, installer can not continue.\n";
            dbg("License not agreed.\n");
            $SIGNAL_UNINSTALL=1;
            return 0;
        }
    } 
    # non-interactive install
    # ( 1 == $autoinstall )
    else
    {
        print_license( ! $autoinstall );
        print "Auto-accepted license because of --autoinstall flag.\n";
    }
    if($licensedto ne "") { $ENV{"DW_INSTALL_LICENSED_TO"} = $licensedto; } # TODO : ??

    ( $answer, $license_file ) = validate_with_license_or_serial($USER_LICENSE_FILE, $LOCAL_LICENSE_FILE, $LICENSE_DESTINATION, $serial, $autoinstall);
    # validated ok ?
    return 0 unless $answer;
    return 1;
} # sub check_license

sub has_portal_account
{
    system("$QXMLEDIT -f $INIFILE --test_portal_account");
    if ($? == 0) {
        return 1;
    }
    return 0;
}

sub init_portal_account
{
    my $username = undef;
    my $password = undef;
    my $confirm_password = undef;
    
    while(1) {
        my $answer = input_confirmation("Do you have a Netop Portal account ?");
        $answer = lc substr($answer,0,1);        
        
        if ( $answer eq 'n') {           
            return 1;
        }
        if ( $answer eq 'q') {   
            return 0; #quit
        }
        
        if ( $answer ne 'y') {   
            next;
        }
        
        $username = ask_for_input("Username");
        if (!$username) { next; }
        $password = ask_for_password_input("Password");
        if (!$password) { next; }        
        $confirm_password = ask_for_password_input("Confirm Password");
        if (!$confirm_password) { next; }       
        
        if ($password ne $confirm_password) {
            print "\nPasswords don't match\n";
            next;
        }
        
        last;
    }
        
    $password = quotemeta($password);
    $username = quotemeta($username);
    
    system("$QXMLEDIT -f $INIFILE --portal_password $password --portal_username $username");
    
    return 1;
}

sub install_chkconfig
{
    $cmd = "$CHKCONFIG_CMD --add netophostd >/dev/null";
    dbg("cmd = $cmd\n");
    my $rc = system($cmd);
    return 0 if($rc != 0);

    if(lc $SYSTEM eq "redhat") {
          $cmd = "$CHKCONFIG_CMD --level 3 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          my $rc = system($cmd);
          return 0 if ($rc != 0);
          $cmd = "$CHKCONFIG_CMD --level 5 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          $rc = system($cmd);
          return 0 if ($rc != 0);

          $UNCONFIGURE_COMMANDS .= "    \`$CHKCONFIG_CMD --del netophostd\`;\n";
          return 1;
    }
    if(lc $SYSTEM eq "suse") {
          $cmd = "$CHKCONFIG_CMD netophostd 35 >/dev/null";
          dbg("cmd = $cmd\n");
          my $rc = system($cmd);
          return 0 if ($rc != 0);
          $UNCONFIGURE_COMMANDS .= "    \`$CHKCONFIG_CMD --del netophostd >/dev/null\`;\n";
          return 1;
    }
    return 0;
}

sub disable_chkconfig
{
    $cmd = "$CHKCONFIG_CMD netophostd off >/dev/null";
    dbg("cmd = $cmd\n");
    my $rc = system($cmd);
    return 0 if($rc != 0);
    return 1;
}

sub query_chkconfig
{
    my $cmd = "$CHKCONFIG_CMD --list netophostd 2>/dev/null";
    dbg("query_chkconfig: cmd: $cmd\n");
    my $output = `$cmd`;
    dbg("query_chkconfig: output: $output\n");
    if( lc $output =~ m|3[:]on|) {
    print "$QUERY_RES 1";
    return 1;
    } else {
    print "$QUERY_RES 0";
    return 1;
    }

    return 0;
}

sub link_script
{
    my ($service_dir)= @_;
    if( -e "$service_dir/S95netophostd") {
    unlink "$service_dir/S95netophostd";
    }
    if( -e "$service_dir/K05netophostd") {
    unlink "$service_dir/K05netophostd";
    }
    dbg("Creating service link scripts in $service_dir\n");
    my $cmd = "$LN_CMD -s $INITD_NETOPHOSTD $service_dir/S95netophostd";
    dbg("cmd = $cmd\n");
    system($cmd);
    $cmd = "$LN_CMD -s $INITD_NETOPHOSTD $service_dir/K05netophostd";
    dbg("cmd = $cmd\n");
    system($cmd);

    if( -e "$service_dir/S95netophostd") {
    $UNCONFIGURE_COMMANDS .= "unlink \"$service_dir/S95netophostd\";\n";
    }
    if( -e "$service_dir/K05netophostd") {
    $UNCONFIGURE_COMMANDS .= "unlink \"$service_dir/K05netophostd\";\n";
    }
    return 1;
}

sub install_links()
{
    my $service_dir = "/etc/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
        if($SYSTEM eq "solaris") {
            return 1;
        }
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    }

    $service_dir = "/etc/init.d/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    }

    $service_dir = "/etc/rc.d/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    } else {
        print "Warning: automatic configuration of runlevels  failed.\n";
        print "Please update relevant runlevel (2,3 or 5) manually.\n";
    return 0;
    }
}

sub disable_link_script
{
    my ($service_dir)= @_;
    unlink "$service_dir/S95netophostd";
    unlink "$service_dir/K05netophostd";
    return 1;
}

sub disable_links()
{
    my $service_dir = "/etc/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    }

    $service_dir = "/etc/init.d/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    }

    $service_dir = "/etc/rc.d/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    } else {
    print "Warning: automatic configuration of runlevel 3 failed.\n";
    print "Please update runlevel 3 manually.\n";
    return 0;
    }
}

sub query_links()
{
    my $service_dir = "/etc/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    return 1;
    }

    $service_dir = "/etc/init.d/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    }

    $service_dir = "/etc/rc.d/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    return 1;
    }

    print "$QUERY_RES 0";
    return 0;
}

sub copy_startup()
{
    # cp platform specific start script to /etc/init.d
    unlink $INITD_NETOPHOSTD;

    if(lc $SYSTEM eq "redhat") {
          $cmd = "$D2U_CMD -n /etc/NetopHost/cfg/initd.rh $INITD_NETOPHOSTD";
    } elsif (lc $SYSTEM eq "suse") {
          $cmd = "$D2U_CMD -n /etc/NetopHost/cfg/initd.suse $INITD_NETOPHOSTD";
    } elsif (lc $SYSTEM eq "ubuntu") {
          $cmd = "$CAT_CMD /etc/NetopHost/cfg/initd.ubuntu | $D2U_CMD > $INITD_NETOPHOSTD";
    }else {
          # includes "debian"
          $cmd = "$CAT_CMD /etc/NetopHost/cfg/initd.gen | $D2U_CMD > $INITD_NETOPHOSTD";
    }

    dbg("cmd = $cmd\n");
    system($cmd);
    system("$CHMOD_CMD +x $INITD_NETOPHOSTD");
    if( ! -f $INITD_NETOPHOSTD) {
          die "Cannot create $INITD_NETOPHOSTD\n";
    }
    return 1;
}

sub enable_startup
{
    my $cmd="";
    dbg("Enable startup script in /etc/init.d !\n");

    my ($output_query_res)= @_;
    # enable startup at runlevel 3 and 5
    if($CHKCONFIG_CMD ne "") {
    if(!install_chkconfig()) {
        goto install_links;
    }
    $UNCONFIGURE_COMMANDS .= "unlink \"$INITD_NETOPHOSTD\";\n";
    if($output_query_res != 0) {
        print "$QUERY_RES 1";
    }
    return 1;
    }

install_links:
    install_links();
    $UNCONFIGURE_COMMANDS .= "unlink \"$INITD_NETOPHOSTD\";\n";
    if($output_query_res != 0) {
    print "$QUERY_RES 1";
    }

    return 1;
}

sub disable_startup()
{
    my $cmd="";
    dbg("Disable startup script in /etc/init.d !\n");

    # disable startup at runlevel 3 and 5
    if($CHKCONFIG_CMD ne "") {
    if(!disable_chkconfig()) {
        goto disable_links;
    }
    print "$QUERY_RES 0";
    return 1;
    }

disable_links:
    disable_links();
    print "$QUERY_RES 0";

    return 1;
}

sub query_startup()
{
    my $cmd="";

    # disable startup at runlevel 3 and 5
    if($CHKCONFIG_CMD ne "") {
    if(!query_chkconfig()) {
        goto query_links;
    }
    return 1;
    }

query_links:
    query_links();

    return 1;
}
# NetopHostXd begin
#
# locate kdm config file
#
sub kdm_config_file
{
     my @trydir_list = ("/etc","/opt");
    my $trydir="";
    my @kdmrc_list=("/etc/kde/kdm/kdmrc", 
                    "/etc/X11/xdm/kdmrc",
            "/etc/opt/kde3/share/config/kdm/Xservers"
            );
    my $kdmrc="";
    my $cmd="";
    my $xservers="";

    while($kdmrc = shift @kdmrc_list) {
          $kdmrc =~ s/\n//g;
          if( ! -e $kdmrc) {
                goto next_kdmrc;
          }
          dbg("From list, testing kdmrc: $kdmrc\n");
          $cmd = "$CAT_CMD $kdmrc";
          dbg("cmd: $cmd\n");
          my $file=`$cmd`;
          if($file =~ m|Xservers[ \t]*=[ \t]*(.+)|) {
                $xservers = $1;
                dbg("Possible Xservers: $xservers\n");
                if( -e $xservers) {
                     return $xservers;
                }
          }
        next_kdmrc:;
    }

    if($FIND_CMD eq "") {
        goto use_list;
    }
    while($trydir = shift @trydir_list) {
          dbg("trydir: $trydir\n");
          if(! -d $trydir) {
                goto next;
          }
          $cmd = "$FIND_CMD $trydir 2>/dev/null | grep kdmrc";
          dbg("cmd: $cmd\n");
          @kdmrc_list = `$cmd`;
          while($kdmrc = shift @kdmrc_list) {
                $kdmrc =~ s/\n//g;
                dbg("Possible kdmrc: $kdmrc\n");
                $cmd = "$CAT_CMD $kdmrc";
                dbg("cmd: $cmd\n");
                my $file=`$cmd`;
                if($file =~ m|Xservers[ \t]*=[ \t]*(.+)|) {
                     $xservers = $1;
                     dbg("Possible Xservers: $xservers\n");
                     if( -e $xservers) {
                          return $xservers;
                     }
                }
          }
        next:;
    }
  use_list:;
    my @configdirs=("/etc/X11/xdm",
                          "/usr/X11R6/lib/X11/xdm",
                          "/etc/kde2/kdm",
                          "/etc/opt/kde3/share/config/kdm",
                          "/opt/kde/share/config/kdm"
                          );
    while(my $xdir = shift @configdirs) {
          $xservers = "$xdir/Xservers";
          dbg("try from list: $xservers\n");
          if( -e $xservers) {
                return $xservers;
          }
    }

    return "";
}

#
# locate gdm config file
#
sub gdm_config_file
{
    my $try="/etc/X11/gdm/gdm.conf";
    if(-e $try) {
          return $try;
    }
    $try="/etc/gdm/custom.conf";
    if(-e $try) {
                  return $try;
    }
    return "";
}

#
# locate dtlogin config file
#
sub dtlogin_config_file
{
    my $try="/usr/dt/config/Xservers";
    if(-e $try) {
          return $try;
    }
    return "";
}


# 0 = not ok
# 1 = X-server configured
# 2 = NetopHostXd already configured
sub check_Xservers_file
{
    my $status = 0;
    my ($file) = @_;
    dbg("check_Xservers_file: $file\n");
    if (! -e $file) {
          return 0;
    }
    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    open INFILE,"<$file" || return 0;
    while ($line = <INFILE>) {
    if ($line =~ m|^[ \t]*:0(.*)|) {
        dbg("possible $line");
        # check if already configured.
        my $right = $1;
        if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
        $status = 2;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
        $status = 2;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X\s*$|m) {
        $status = 1;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
    }
    }
  closedown:
    close INFILE;
     dbg("check_Xservers_file: returning $status\n");
    return $status;
}

# 0 = not ok
# 1 = X-server configured    (wrong)
# 2 = NetOpHostXd configured (ok)
sub install_Xservers_file
{
    my $status = 0;
    my ($file) = @_;
    dbg("install_Xservers_file: $file\n");
    if(! -e $file) {
          return 0;
    }
    # backup
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
          dbg("backup file failed\n");
          return 0;
    }
    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open";
    if($? != 0) {
          dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
          system("ls -l $infile");
          return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
          dbg("open OUTFILE failed ($?): $outfile\n");
          close INFILE;
          return 0;
    }
    while($line = <INFILE>) {
    if($line =~ m|^[ \t]*:0(.*)|) {
        my $right = $1;
        if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
        print OUTFILE $line;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
        print OUTFILE $line;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
    } else {
        print OUTFILE $line;
    }
    }
  closedown:
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_Xservers_file($file);
    if ($status != 2) {
          # hm. not updated, restore previous
          dbg("check_Xservers_file: failed, restoring backup\n");
          $cmd = "$CAT_CMD < $file.netop > $file";
          dbg("cmd: $cmd\n");
          system("$cmd");
    }
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    return $status;
}


sub real_Xserver
{
    my $Xpath = `which X`;
    chomp $Xpath;

    my @list = ("/usr/X11R6/bin/X", "/usr/openwin/bin/Xsun", "/usr/X11/bin/Xserver", "$Xpath");
    while (my $server = shift(@list)) {
#   print "MLU server: $server\n";
    if (-e $server) {
        return $server;
    }
    }
    return "";
}


# 0 = not ok
# 1 = X-server configured    (ok)
# 2 = NetOpHostXd configured (wrong)
sub uninstall_Xservers_file
{
    my $status = 0;
    my ($file) = @_;

    dbg("uninstall_Xservers_file: $file\n");
    if(! -e $file) {
          return 0;
    }
    my $real_xserver = real_Xserver();
    dbg("real_xserver = $real_xserver\n");
    if($real_xserver eq "") {
          return 0;
    }
    # backup
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
          dbg("backup file failed\n");
          return 0;
    }

    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    my $infile = "$file.netop";
    open INFILE,"<$infile";
    if($? != 0) {
          dbg("uninstall_Xservers_file: open INFILE failed ($?): $infile\n");
          system("ls -l $infile");
          return 0;
    }

    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
          dbg("open OUTFILE failed ($?): $outfile\n");
          close INFILE;
          return 0;
    }
    while ($line = <INFILE>) {
          if ($line =~ m|^[ \t]*:0(.*)|) {
                my $right = $1;
                if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
                     dbg("uninstall_Xservers_file::0$1 $real_xserver\n");
                     print OUTFILE ":0$1 $real_xserver\n";
                }
                elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
                     dbg("uninstall_Xservers_file::0$1 $real_xserver $2\n");
                     print OUTFILE ":0$1 $real_xserver $2\n";
                }
          } else {
                print OUTFILE $line;
          }
    }
    closedown:
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_Xservers_file($file);
    if ($status != 1) {
          # hm. not updated, restore previous
          dbg("check_Xservers_file: failed, restoring backup\n");
          $cmd = "$CAT_CMD < $file.netop > $file";
          dbg("cmd: $cmd\n");
          system("$cmd");
    }
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub check_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if (! -e $file) {
        return 0;
    }
    open INFILE,"<$file" || return 0;
    while (my $line = <INFILE>) {
        if ($state == 0) {
            # locate section [servers]
            $line = lc $line;
            if ($line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
        } elsif ($state == 1) {
            # locate "0=server"
            if($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                my $server = $1;
                dbg("possible server '$server'\n");
                if ($server eq "/usr/sbin/netophostxd") {
                    $status = 2;
                    goto closedown;
                }
                $status = 1;
                goto closedown;
            }
        }
    }
    closedown:;
    close INFILE;
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub install_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if(! -e $file) {
        return 0;
    }
    # backup
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
        dbg("backup file failed\n");
        return 0;
    }
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open $infile";
    if($? != 0) {
        dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
        system("ls -l $infile");
        return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
        dbg("open OUTFILE failed ($?): $outfile\n");
        close INFILE;
        return 0;
    }
    while(my $line = <INFILE>) {
        if($state == 0) {
            # locate section [servers]
            $tst_line = lc $line;
            if($tst_line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
            print OUTFILE $line;
        } elsif ($state == 1) {
            # locate "0=server"
            if($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                my $server = $1;
                dbg("possible server '$server'\n");
                if($server eq $NETOPHOSTXD) {
                    print OUTFILE $line;
                } else {
                    # backup existing line
                    print OUTFILE "#netop_save $line";
                    # TBD: options
                    print OUTFILE "0=$NETOPHOSTXD\n";
                }
                $state = 2;
            }
        } else {
            print OUTFILE $line;
        }
    }
    closedown:;
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_gdm_file($file);
    if($status != 2) {
         # hm. not updated, restore previous
         dbg("check_gdm_file: failed, restoring backup\n");
         $cmd = "$CAT_CMD < $file.netop > $file";
         dbg("cmd: $cmd\n");
         system("$cmd");
    }
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub uninstall_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if(! -e $file) {
        return 0;
    }
    # backup
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
        dbg("backup file failed\n");
        return 0;
    }
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open $infile";
    if($? != 0) {
        dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
        system("ls -l $infile");
        return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
        dbg("open OUTFILE failed ($?): $outfile\n");
        close INFILE;
        return 0;
    }
    while(my $line = <INFILE>) {
        if($state == 0) {
            # locate section [servers]
            $tst_line = lc $line;
            if($tst_line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
            print OUTFILE $line;
        } elsif ($state == 1) {
            # locate #netop_save or 0=
            # locate "0=server"

            if($line =~ m|^#netop_save (.*)|) {
                $state = 3;
                $server = $1;
                dbg("restored netop_save line: '$server'");
                print OUTFILE "$server\n";
            } elsif ($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                $server = $1;
                dbg("found backed up server line: $server\n");
                if($server =~ $NETOPHOSTXD) {
                    # skip line
                } else {
                    print OUTFILE $line;
                }
                $state = 2;
            }
        } elsif ($state == 3) {
            if ($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                # skip line, netop_save line already inserted
                $state = 4;
            } else {
                print OUTFILE $line;
            }
        } else {
            print OUTFILE $line;
        }
    }
    closedown:;
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_gdm_file($file);
    if($status != 1) {
         # hm. not updated, restore previous
         dbg("check_gdm_file: failed, restoring backup\n");
         $cmd = "$CAT_CMD < $file.netop > $file";
         dbg("cmd: $cmd\n");
         system("$cmd");
    }
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    return $status;
}

my $NEED_RESTART=0;

sub install_netophostxd
{
     my $ret_status=1;
    my $server = $NETOPHOSTXD;
    if (! -e $server) {
        print "install_netophostxd: server not found: $server\n";
        return 0;
    }
    my $Xservers = "";
    if ($SYSTEM eq "solaris") {
          $Xservers = dtlogin_config_file();
    } else {
          $Xservers = kdm_config_file();
    }
    
    print "xserver kdm: $Xservers\n";
    if ($Xservers eq "") {
          goto gdm_install;
    }
    my $status = check_Xservers_file($Xservers);
    if ($status == 0) {
    dbg("check_Xservers_file error\n");
        $ret_status = 0;
          goto gdm_install;
    }
    if ($status == 2) {
          # already installed.
          goto gdm_install;
    }
    $status = install_Xservers_file($Xservers);
    $NEED_RESTART = 1;
    if ($status == 0) {
    dbg("install_Xservers_file error\n");
        $ret_status = 0;
          goto gdm_install;
    }
    if ($status != 2) {
        print "install_netophostxd: problems installing netophostxd in $Xservers\n";
        $ret_status = 0;
          goto gdm_install;
    }
     
  gdm_install:;
    my $gdmconf = gdm_config_file();
    if ($gdmconf eq "") {
        goto fino;
    }
    $status = check_gdm_file($gdmconf);
    if ($status == 0) {
    dbg("check_gdm_file error\n");
        $ret_status = 0;
        goto fino;
    }
    if ($status == 2) {
          # already installed.
        goto fino;
    }
    $status = install_gdm_file($gdmconf);
    $NEED_RESTART = 1;
    if ($status == 0) {
    dbg("install_gdm_file error\n");
        $ret_status = 0;
        goto fino;
    }
    if ($status != 2) {
          print "install_netophostxd: problems installing netophostxd in $gdmconf\n";
        $ret_status = 0;
        goto fino;
    }
  fino:;
    return $ret_status;
}

sub uninstall_netophostxd
{   my $ret_status=1;
    my $server = $NETOPHOSTXD;
    if (! -e $server) {
        print "uninstall_netophostxd: server not found: $server\n";
        return 0;
    }
    my $Xservers = "";
    if ($SYSTEM eq "solaris") {
          $Xservers = dtlogin_config_file();
    } else {
          $Xservers = kdm_config_file();
    }
    if($Xservers eq "") {
        goto gdm_uninstall;
    }
    my $status = check_Xservers_file($Xservers);
    if ($status == 0) {
        $ret_status = 0;
        goto gdm_uninstall;
    }
    if ($status == 1) {
        # already uninstalled.
        goto gdm_uninstall;
    }
    $status = uninstall_Xservers_file($Xservers);
    $NEED_RESTART = 1;
    if($status == 0) {
        $ret_status = 0;
        goto gdm_uninstall;
    }
    if ($status != 1) {
          print "uninstall_netophostxd: problems uninstalling netophostxd in $Xservers\n";
          $ret_status = 0;
          goto gdm_uninstall;
    }

    gdm_uninstall:;
    my $gdmconf = gdm_config_file();
    if($gdmconf eq "") {
        goto fino;
    }
    $status = check_gdm_file($gdmconf);
    if ($status == 0) {
        $ret_status = 0;
        goto fino;
    }
    if ($status == 1) {
        # already uninstalled.
        goto fino;
    }
    $status = uninstall_gdm_file($gdmconf);
    $NEED_RESTART = 1;
    if ($status == 0) {
        $ret_status = 0;
        goto fino;
    }
    if ($status != 1) {
          print "uninstall_netophostxd: problems uninstalling netophostxd in $gdmconf\n";
          $ret_status = 0;
          goto fino;
    }
    fino:;
    return $ret_status;
}
# NetOpHOstXd end



#
# solaris crle config begin
#
my $CRLE_CONFIG  = "/var/ld/ld.config";
my $LDD_BINARY = "/bin/ldd";
my $CRLE_PATH = "";

sub configure_crle()
{
    return 1 if($CRLE_PATH eq "");

    if(-f $CRLE_CONFIG) {
          # file exist, use update flag: -u
          $cmd = "crle -u -l $CRLE_PATH";
    } else {
          # file does not exist, remember to include /usr/lib
          $cmd = "crle -l /usr/lib:$CRLE_PATH";
    }
    dbg("configure_crle: crle_command: '$cmd'\n");
    if( system $cmd ) {
          die "error in $cmd\n";
    }
}

#
# BEGIN ldd test tools
#
$FILE_EXPR  = "";
$SEARCH_DIR = "";
# locate $FILE_EXPR in $SEARCH_DIR
# return 1 if match, 0 if no match
sub find_file()
{
    my @filelist = `/bin/ls $SEARCH_DIR`;
    while ($file = pop @filelist) {
       return 1 if($file =~ m|$FILE_EXPR| );
    }
    return 0;
}

sub append_crle_path()
{
    if($CRLE_PATH eq "") {
          $CRLE_PATH = $SEARCH_DIR;
    } else {
          if(!($CRLE_PATH =~ $SEARCH_DIR)) {
                $CRLE_PATH = "$CRLE_PATH:$SEARCH_DIR";
          }
    }
}

$SO_MISSING_LIST = "";
$MISSING = "";
sub append_so_missing_list()
{
    if($SO_MISSING_LIST eq "") {
    $SO_MISSING_LIST = $MISSING;
    } else {
    if(!($SO_MISSING_LIST =~ $MISSING)) {
        $SO_MISSING_LIST = "$SO_MISSING_LIST, $MISSING";
    }
    }
}

# build directory list for CRLE -> $CRLE_PATH.
# return 0 if all ok.
# return 1 if CRLE path needs updated $CRLE_PATH is updated.
# return 2 if unresolved libraries exist (needs to be installed on system).
sub test_ldd
{
    my $st=0;
    my ($exe) = @_;
    my @ld = `$LDD_BINARY $exe`;
    while ($line = pop @ld) {
       my $WSP = "[ \t]";
       my $NWSP = "[^ \t]";
       my $regexp = "^$WSP*(lib$NWSP+.so$NWSP*)$WSP+=>$WSP+[(][^)]+[)].*\$";
       if($line =~ m|$regexp| ) {
       $missing = $1;
       if($missing =~ m|libXtst.so| ) {
           $SEARCH_DIR = "/usr/openwin/lib";
           $FILE_EXPR  = "libXtst.so";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } elsif ($missing =~ m|libstdc[+][+].so| ) {
           $SEARCH_DIR = "/usr/local/lib";
           $FILE_EXPR  = "libstdc[+][+].so";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } elsif ($missing =~ m|libgcc| ) {
           $SEARCH_DIR = "/usr/local/lib";
           $FILE_EXPR  = "libgcc";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } else {
           if($VERBOSE) {printf "Missing ref to $1 cannot resolve.\n"};
           append_so_missing_list();
           $MISSING = $missing;
           $st = 2 if($st < 2);
       }
       }
    }
    return $st;
}
#
# END ldd test tools
#

#
# Forall executable: ensure that it can be run.
#
sub verify_requirements
{
    my $fatal_missing=0;
    my @exe_filelist = @_;
    while(my $exe = pop @exe_filelist) {
        die "Missing executable: '$exe'\n" if(! -e $exe);
        my $st = test_ldd($exe);
    if($st > $fatal_missing)  { $fatal_missing = 1};
    if($st > $fatal_missing)  { $fatal_missing = 2};
    }

    return $fatal_missing;
}

my $NEED_LD_PATH=0;
sub ensure_solaris_environment
{
    if($SYSTEM ne "solaris") {
        return 1;
    }
    my $ld_library_path = $ENV{"LD_LIBRARY_PATH"};
    my $st = verify_requirements("/usr/sbin/netophost");
    dbg("CRLE_PATH = '$CRLE_PATH'\n");
    # append LD_LIBRARY_PATH to crle if exist.
    if($st == 2) {
         die "Missing dynamic libraries on system: $SO_MISSING_LIST";
    }
    if($st != 0 || ($CRLE_PATH ne "")) {
        configure_crle();
    }
    if(verify_requirements("/usr/sbin/netophost")) {
        my $VER="8";
        if($OS_VER eq "5.9") {
            $VER = "9";
        }
        if(! -d "/usr/local/lib" || ! -e "/usr/local/lib/libstdc++.so.5") {
            print "! \n";
            print "! Warning: required library: libstdc++.so.5 is not installed on the system.\n";
            print "! libstdc++.so.5 is located in package SMClibgcc.\n";
            print "! SMClibgcc can be downloaded from www.sunfreeware.com\n";
            print "! or from any of the mirror sites.\n";
            print "! (Download the file: libgcc-3.2.X-sol$VER-sparc-local)\n";
            print "! \n";
            print "! Install using pgkadd:\n";
            print "! pkgadd -d libgcc-3.2-sol$VER-sparc-local\n";
            print "! \n";
            return 0;
        }
        print "! Warning:\n"
             ."! Cannot configure Runtime Environment (crle), please configure manually.\n"
             ."! info: 'ldd /usr/sbin/netophost' shows unresolved libraries.\n"
             ."! end Warning.\n";
        $NEED_LD_PATH=1;
        return 0;
    }
    return 1;
}
#
# solaris crle config end
#

sub su_test
{
    if(! -e $ID_CMD) {
        return 1; # continue, cannot get uid.
    }
    my $id = `$ID_CMD`;
    if($id =~ m|uid=([0-9]+)|) {
        if($1 eq "0") {
            return 1;
        }
    }
    return 0;
}

sub setup_pam()
{
# 	if ($OS_BIT =~ "i686"){
# 		return;
#     }
    my $prefix = "";
    if(-e "/lib32/security/pam_unix.so"){
        $prefix = "/lib32/security/";
    }
    if (!open(PAMCONF, ">/etc/pam.d/netophost")) {
        print "install pam configuration file: Unable to create configuration file\n";
        return 0;
    }
    print PAMCONF "#%PAM-1.0\n";
    print PAMCONF "auth       sufficient       ".$prefix."pam_unix2.so shadow nullok\n";
    print PAMCONF "auth       sufficient       ".$prefix."pam_unix.so shadow nullok\n";
    print PAMCONF "auth       required         ".$prefix."pam_unix_auth.so shadow nullok\n";
    print PAMCONF "account    sufficient       ".$prefix."pam_unix2.so\n";
    print PAMCONF "account    sufficient       ".$prefix."pam_unix.so\n";
    print PAMCONF "account    required         ".$prefix."pam_unix_acct.so\n";
    close(PAMCONF);
}

sub setup_logrotate()
{
    $logrotated_dir = "/etc/logrotate.d";
    if( ! -d $logrotated_dir ) {
        system("/usr/bin/mkdir -p $logrotated_dir");
        if( ! -d $logrotated_dir ) {
            die "Cannot create $logrotated_dir\n";
        }
    }

    $logrotate_src = "/etc/NetopHost/cfg/logrotate";
    $logrotate_dst = "$logrotated_dir/netophost";
    if(lc "$D2U_CMD" eq "dos2unix") {
        $cmd = "$D2U_CMD -n $logrotate_src $logrotate_dst >/dev/null";
    } else {
        # assumes fromdos
        $cmd = "$CAT_CMD $logrotate_src | $D2U_CMD > $logrotate_dst";
    }
    dbg($cmd);
    system($cmd);
    if( ($? == -1) or ($? >> 8 != 0) ) {
        die "Cannot copy $logrotate_src to $logrotate_dst\n";
    }

    $UNCONFIGURE_COMMANDS .= "\n";
    $UNCONFIGURE_COMMANDS .= "# unregister log rotation\n";
    $UNCONFIGURE_COMMANDS .= "unlink \"$logrotated_dir/netophost\n\";";
    $UNCONFIGURE_COMMANDS .= "\n";
}

sub main()
{
    my $serial      = undef;
    my $autoinstall = 0;
    my $noinitd     = 0;
    my $licensedto  = "";
    my $usage =  "Usage: configure.pl <param>\n"
          ." where <param> is:\n"
          ." installation:\n"
          ."    [--autoinstall] : non-interactive, do not ask questions.\n"
          ."    [--noinitd]     : do not add service startup script to /etc/init.d\n"
          ."    [--serial    <serial>] : license serial (UK00900-XXXX...). \n"
          ."    or:\n"
          ."    [--install_netophostxd]   : configure netophostxd and exit\n"
          ."    [--uninstall_netophostxd] : unconfigure netophostxd and exit\n"
          ."    [--enable_startup]        : enable runlevel startup and exit\n"
          ."    [--disable_startup]       : disable runlevel startup and exit\n"
          ."    [--query_startup]         : print runlevel startup info and exit\n"
          ."    [--enable_crle]           : (Solaris) ensure correct crle and exit.\n"
          ."    [--help]                  : print help message and exit.\n"
          ."    [--version]               : print version info and exit.\n";
    while ($arg = shift @ARGV) {
          if ($arg eq "--version") {
                print "$VERSION\n";
                exit;
          } elsif ($arg eq "--autoinstall") {
                $autoinstall = 1;
          } elsif ($arg eq "--install_netophostxd") {
            die "Must be root " if(! su_test());
                system_setup();
                setup_tools();
            die "Could not install netophostxd" if (($HOSTXD_WORKS eq "yes") and !install_netophostxd());
            print "-oOo- netophostxd is successfully configured -oOo-\n";
            if($NEED_RESTART != 0) {
                print "-oOo- Note: display manager needs to be restarted (or machine rebooted). -oOo-\n";
                print "-oOo-       for the changes to take full effect.                         -oOo-\n";
            }
            exit;
          } elsif ($arg eq "--uninstall_netophostxd") {
            die "Must be root " if(! su_test());
                system_setup();
                setup_tools();
            die "Could not uninstall netophostxd" if (($HOSTXD_WORKS eq "yes") and !uninstall_netophostxd());
            print "-oOo- netophostxd is successfully unconfigured -oOo-\n";
            if($NEED_RESTART != 0) {
                print "-oOo- Note: display manager needs to be restarted (or machine rebooted). -oOo-\n";
                print "-oOo-       for the changes to take full effect.                         -oOo-\n";
            }
            exit;
          } elsif ($arg eq "--noinitd") {
                $noinitd = 1;
          } elsif ($arg eq "--help") {
                print $usage;
                exit;
          } elsif ($arg eq "--serial") {
                # next token is not another option ?
                 if ( ($ARGV[0]) and ( "--" ne substr($ARGV[0],0,2) ) ) { $serial = shift @ARGV; } 
          } elsif ($arg eq "--enable_startup") {
            die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                enable_startup(1);
                return 0;
        } elsif ($arg eq "--enable_crle") {
            die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
            ensure_solaris_environment();
                return 0;
          } elsif ($arg eq "--disable_startup") {
                die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                disable_startup();
                return 0;
          } elsif ($arg eq "--query_startup") {
                die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                query_startup();
                return 0;
          } elsif ($arg eq "--pam") {
                setup_pam();
                return 0;
          } else {
                print "Error: unknown arg: '$arg'\n";
                die $usage;
          }
    }
    die "Must be root " if (!su_test());

    $UNCONFIGURE_COMMANDS  = "#!/usr/bin/perl -w\n";
    $UNCONFIGURE_COMMANDS .= "# unconfigure script for Netop Host\n";
    $UNCONFIGURE_COMMANDS .= "if(-e \"$INITD_NETOPHOSTD\"){\n"; # BEGIN BLOCK 1
    $UNCONFIGURE_COMMANDS .= "    \`$INITD_NETOPHOSTD stop\`;\n";

    system_setup();
    setup_tools();

    if ($SYSTEM eq "solaris") {
        if(!ensure_solaris_environment()) {
            exit 1;
        }
    }

    if (!check_license($serial,$licensedto,$autoinstall)) {
        #die "Licence check failed, Netop Host not operational.\n";
        host_installed_but_not_validated;
        
        if ($SIGNAL_UNINSTALL != 0) {
            $! = 56;
        }
        exit (0);
    }
    
    
    if (!$autoinstall && !has_portal_account() && !init_portal_account()) {
        exit(0);
    }
    

    if (($HOSTXD_WORKS eq "yes") and !install_netophostxd()) {
        print STDERR "Installation of netophostxd failed, please install manually.\n";
    }

    if (!$noinitd) {
        copy_startup();
        enable_startup(0);
        system("cd /;$INITD_NETOPHOSTD start");
    }

	setup_pam();

    $UNCONFIGURE_COMMANDS .= "}\n";  # END BLOCK 1
    $UNCONFIGURE_COMMANDS .= "`perl /etc/NetopHost/cfg/configure.pl --uninstall_netophostxd`;\n";

    setup_logrotate();

    $UNCONFIGURE_COMMANDS .= "unlink \"/etc/NetopHost/cfg/unconfigure.pl\";\n";
    #$UNCONFIGURE_COMMANDS .= "unlink \"$LICFILE\";\n";
    $UNCONFIGURE_COMMANDS .= "print \"Netop Host successfully unconfigured\\n\";\n";
     
    write_unconfigure();
    chmod 0744,$UNCONFIGURE_SCRIPT;
    dbg("Unconfigure script is written to $UNCONFIGURE_SCRIPT\n");
    host_installed_and_validated( $NEED_RESTART ); 
}

main;
exit;
__END__
__LICENSE_FOLLOWS__
Netop Business Solutions A/S End-User License Agreement 
Updated: October 2014

This Agreement provides you with the right to install, load, host and use “the Software” as described herein.  

Portions of the Software licensed under this Agreement are used under license from third parties. All Rights Reserved. 

NOTICE TO ALL USERS: PLEASE READ THE FOLLOWING LEGAL AGREEMENT (“AGREEMENT“) CAREFULLY. THIS AGREEMENT CONSTITUTES A LEGAL, BINDING AND ENFORCEABLE AGREEMENT BETWEEN YOU AND NETOP BUSINESS SOLUTIONS A/S (“NETOP BUSINESS SOLUTIONS”) REGARDING ALL USE OF ANY NETOP BUSINESS SOLUTIONS SOFTWARE (“SOFTWARE“). BY INSTALLING THE SOFTWARE, YOU (EITHER AN INDIVIDUAL OR A SINGLE ENTITY) ACCEPT TO BE BOUND BY AND BECOME A PARTY TO THIS AGREEMENT WITH NETOP BUSINESS SOLUTIONS. IF YOU DO NOT AGREE TO ALL OF THESE TERMS, DO NOT INSTALL OR USE THE SOFTWARE. 

1. License Grant. Subject to the payment of the applicable license fees, and subject to your abidance by the terms and conditions of this Agreement, Netop Business Solutions hereby grants to you a non-exclusive, non-transferable right to use one copy of the specified version of the Software which you have acquired and the accompanying documentation (the “Documentation“). 

2.  Use. The use of products is described in appendices which is an integrated part of this End User Licenses Agreement and thus legally binding on You. Please find the proper appendix according to your product type which is stated on your invoice or order confirmation.  

3. Term. This Agreement is effective until terminated. You may terminate it at any point by destroying or deleting the Software together with all copies of the Software. Also, Netop Business Solutions has the option to terminate this Agreement if you fail to comply with any terms of this Agreement. You agree upon such termination immediately to destroy or deleting all parts of the Software and the Documentation together with all copies thereof. 

4. Updates. Netop Business Solutions may update the content of the Software from time to time and provided that you have purchased the right to upgrades, you may download such revisions, upgrades, or updates to the Software when and as Netop Business Solutions publishes them via its website or other online services. You agree that the terms of this Agreement (or any subsequent version hereof) shall applies to any updated version of the Software or the Documentation. 

5. Ownership Rights. International copyright laws and international treaty provisions protect the Software. Netop Business Solutions and its suppliers own and retain all right, title and interest in and to the Software and the Documentation, including all copyrights, patents, trade secret rights, trademarks and other intellectual property rights therein. Your possession, installation, or use of the Software does not transfer to you any title to the intellectual property in the Software, and you will not acquire any rights to the Software except as expressly set forth in this Agreement. All copies of the Software and Documentation made hereunder will contain the same proprietary notices that appear on and in the Software and the Documentation. 

6. Restrictions. You may not give away, rent, lease, loan or resell the Software and the Documentation or transfer any of the rights granted to you under this Agreement. You may not reverse engineer, decompile, hack, translate or disassemble the Software (except to the exact extent that applicable law expressly prohibits the foregoing restriction). You may not modify, or create derivative works based upon the Software in whole or in part. You may not copy the Software or Documentation except as expressly permitted in Section 1 above. You may not remove any proprietary notices or labels on the Software or Documentation. Nor may you cause or permit others to transact any of the actions which are governed by this Section 6. Netop Business Solutions reserves all rights not expressly set forth hereunder. 

7. Warranty and Disclaimer. 

a. Limited Warranty. Netop Business Solutions warrants that for sixty (60) days from the date of original purchase, the media (for example, the diskettes) on which the Software is contained will be free from defects in materials and workmanship.

b. Customer Remedies. Netop Business Solutions and its suppliers’ entire liability, and your exclusive remedy for any breach of the foregoing warranty, shall be, at Netop Business Solutions’ option, either (i) to return the purchase price paid for the license, if any, or (ii) to replace the defective media on which the Software is contained. You must return the defective media to Netop Business Solutions at your expense with a copy of your receipt. This limited warranty is void if the defect has resulted from accident, abuse, or misapplication. Any replacement media will be warranted for the remainder of the original warranty period. This remedy is not available to the extent that Netop Business Solutions is subject to restrictions under export control laws and regulations.

c. Warranty Disclaimer. THE SOFTWARE IS PROVIDED “AS IS“ and “AS AVAILABLE“. TO THE MAXIMUM EXTENT PERMITTED BY APPLICABLE DANISH LAW, NETOP BUSINESS SOLUTIONS DISCLAIMS ALL WARRANTIES, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NONINFRINGEMENT WITH RESPECT TO THE SOFTWARE AND THE ACCOMPANYING DOCUMENTATION. YOU ASSUME RESPONSIBILITY FOR SELECTING THE SOFTWARE TO ACHIEVE YOUR INTENDED RESULTS, AND FOR THE INSTALLATION OF, USE OF, AND RESULTS OBTAINED FROM THE SOFTWARE. WITHOUT LIMITING THE FOREGOING PROVISIONS, NETOP BUSINESS SOLUTIONS UNDERTAKES NO LIABILITY THAT THE SOFTWARE WILL BE ERROR-FREE OR FREE FROM INTERRUPTIONS, PROBLEMS CO-WORKING WITH OTHER SOFTWARE PACKAGES OR OPERATING SYSTEMS, OR ANY OTHER FAILURES, OR THAT THE SOFTWARE WILL MEET YOUR REQUIREMENTS, WHETHER EXPRESSLY ACKNOWLEDGED OR NOT.

d. Limitation of Liability. UNDER NO CIRCUMSTANCES AND UNDER NO LEGAL THEORY, WHETHER IN TORT, CONTRACT, OR OTHERWISE, SHALL NETOP BUSINESS SOLUTIONS OR ITS SUPPLIERS BE LIABLE TO YOU OR TO ANY OTHER PERSON (PHYSICAL OR LEGAL) FOR ANY INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES OF ANY CHARACTER INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF GOODWILL, WORK STOPPAGE, COMPUTER FAILURE, LOSS OF DATA OR MALFUNCTION, OR FOR ANY AND ALL OTHER DAMAGES OR LOSSES. IN NO EVENT WILL NETOP BUSINESS SOLUTIONS BE LIABLE FOR ANY DAMAGES IN EXCESS OF THE LIST PRICE NETOP BUSINESS SOLUTIONS CHARGES FOR A LICENSE TO THE SOFTWARE, EVEN IF NETOP BUSINESS SOLUTIONS SHALL HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. THIS LIMITATION OF LIABILITY SHALL NOT APPLY TO LIABILITY FOR DEATH OR PERSONAL INJURY TO THE EXTENT ONLY THAT APPLICABLE LAW PROHIBITS SUCH LIMITATION. The foregoing provisions shall be enforceable to the maximum extent permitted by applicable law. 

8. High Risk Activities. The Software is not fault-tolerant and is not designed or intended for use in hazardous environments that require fail-safe performance, including without limitation, in the operation of nuclear facilities, aircraft navigation or communication systems, air traffic control, weapons systems, direct life-support machines, or any other application in which the failure of the Software could lead directly to death, personal injury, or severe physical or property damage collectively, “High Risk Activities“. Netop Business Solutions expressly disclaims any express or implied warranty of fitness for High Risk Activities. 

9. Miscellaneous. This Agreement is governed by the laws of Denmark, without reference to conflict of law principles. The application of the United Nations Convention of Contracts for the International Sale of Goods is expressly excluded. This Agreement sets forth all rights for the user of the Software and is the entire agreement between the parties. This Agreement supersedes any other communications with respect to the Software and Documentation. This Agreement may not be modified except by a written addendum issued by Netop Business Solutions. No provision hereof shall be deemed waived unless such waiver shall be in writing and signed by Netop Business Solutions. If any provision of this Agreement is held invalid, the remainder of this Agreement shall continue in full force and effect. The provision, which is held invalid, shall to the furthest extent possible be legally interpreted and construed in accordance with the intentions and the nature of this entire Agreement. The parties confirm that it is their wish that this Agreement has been written in the English language only. Nothing contained in this section 9 shall be construed as an obligation by you to disclose any of your proprietary or confidential information to any third party. 

10. Your representations and warranties. You represent and warrant that you are authorized to enter into this Agreement and comply with its terms. Furthermore, you represent and warrant that you will at any and all times comply with your obligations hereunder and any applicable laws, regulations and policies, which may apply to the Software. You assume full and unlimited liability for any use contrary to this entire Agreement whether such use has been enacted or caused directly or indirectly by you. You agree to indemnify, defend and hold Netop Business Solutions harmless from and against any and all liability arising out of or in connection with your violation or breach of this Agreement or applicable laws, regulations and policies, or Your negligence or willful misconduct 

11. NETOP BUSINESS SOLUTIONS CUSTOMER CONTACTS. If you have any questions concerning these terms and conditions, or if you would like to contact Netop Business Solutions for any other reason, please write to Netop Business Solutions A/S, Bregnerodvej 127, DK-3460 Birkerod, Denmark or visit our website www.netop.com for detailed contact information. 
 
Netop Business Solutions A/S End-User License Agreement Appendices

Appendix A: Products purchased as perpetual licenses

Use of perpetual licenses: The Software is licensed as a single product; it may not be used on more than one Client Device or by more than one user at a time (unless Your License Certificate particularly authorizes such use). The Software is “in use“ on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or installed into the permanent memory (e.g., hard disk, CD-ROM, or other storage device) of that Client Device. This license authorizes You to make one copy of the Software solely for backup or archival purposes, provided that the copy You make contains all of the proprietary notices for the Software.


1. Server Use. To the extent that the applicable order acknowledgement, product invoice, product packaging or License Certificate sets forth, you may use the Software on a Client Device or on a Server within a multi-user or networked environment (“Server Use“). A separate license is required for each Client Device or “seat“ that may connect to the Software at any time, regardless of whether such licensed Client Devices or seats are connected to the Software concurrently, or are actually using the Software at any particular time. Your use of software or hardware that reduces the number of Client Devices or seats that connect to and use the Software directly or simultaneously (e.g., “multiplexing“ or “pooling“ software or hardware) does not reduce the number of licenses required. Specifically, You must have that number of licenses that would equal the number of distinct inputs to the multiplexing or pooling software or hardware “front end“). If the number of Client Devices or seats that can connect to the Software can exceed the number of licenses You have obtained, then You must have a reasonable mechanism in place to ensure that Your use of the Software does not exceed the use limits specified for the license You have obtained. This license authorizes You to make or download one copy of the Documentation for each Client Device or seat that is licensed, provided that each such copy contains all of the proprietary notices for the Documentation.

2. Volume Use. If the Software is licensed with volume use terms specified in the applicable order acknowledgement, product invoice, License Certificate or product packaging, You may make, install and use as many additional copies of the Software on the number of Client Devices as the terms specify. You must have a reasonable mechanism in place to ensure that the number of Client Devices on which the Software has been installed does not exceed the number of licenses you have obtained. This license authorizes You to make or download one copy of the Documentation for each additional copy authorized by the volume license, provided that each such copy contains all of the proprietary notices for the Documentation.

3. Concurrent Use. If the Software is licensed with Concurrent Use terms as specified in the applicable order acknowledgement, product invoice, License Certificate or product packaging, You may install the associated Software on an unlimited number of Client Devices .The Software may be used or accessed  on as many simultaneous sessions according to the number of  purchased accesses according to the applicable order acknowledgement, product invoice, License Certificate or product packaging . The Software is “in use“ on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or accessed by You or an employee or authorized agent of You. Should the same employee or authorized agent of You have multiple simultaneous accesses to the Software, each access shall be separately considered a Concurrent User A license for a Concurrent User only grants access rights to individuals; devices and machines may not utilize a Concurrent User license to access to the Software.

4. Vendor signing. For the Android version of Netop Mobile and Embedded it is explicitly prohibited for end-users or any other to vendor sign or cause vendor signing of this module and thereby enabling this module to reach necessary system resources needed for the viewing of the screen and injection of input to the devices, that it otherwise could not reach. Doing so immediately revokes the license to use this module.

5. Services and features depending on Netop Advantage Annual Service & Upgrade Program.  Vision Pro Learning Center is a feature running as a service (SaaS). The feature is included with Vision Pro for as long as You have an active Netop Advantage program and continuous use of Learning Center Service is dependent on Your subscription to the Netop Advantage Program. If Your Netop Advantage subscription expires You will no longer be able to access and use Vision Pro Learning Center. Use of the Learning Center feature is subject and regulated by this agreement appendix C.  Netop will make storage space available in the Vision Pro Learning Center. Any storage space made available by Netop will have limitations on such storage space as set by Netop from time to time, and the available storage space can be subject to change by Netop without your consent given no less than 30 days notice.  

Netop Advantage Annual Service & Upgrade Program: Your software license includes 12 months of free product service, support and software upgrade(s). After 12 months a renewal of Netop Advantage is available for a fee of 20% of the current list price for your software license. A renewal allows you access to Netop support services, resolution of support cases according to Netop’s Service Level Agreement and full investment protection in terms of free software upgrades when available. When Your Netop Advantage subscription expires the above mentioned services will no longer be available. Please refer to www.netop.com/support-overview further details about support services and the Netop Advantage program.

Appendix B: Products purchased as subscriptions

THIS SUBSCRIPTION AGREEMENT (“AGREEMENT”) GOVERNS YOUR FREE TRIAL OF THE PRODUCTS. IF YOU PURCHASE OUR PRODUCTS, THIS AGREEMENT WILL ALSO GOVERN YOUR PURCHASE AND ONGOING USE OF THOSE PRODUCTS. 

BY ACCEPTING USE OF FREE TRIAL OR BY EXECUTING AN ORDER FORM THAT REFERENCES THIS AGREEMENT, YOU AGREE TO THE TERMS OF THIS AGREEMENT. IF YOU ARE ENTERING INTO THIS AGREEMENT ON BEHALF OF A COMPANY OR OTHER LEGAL ENTITY, YOU REPRESENT THAT YOU HAVE THE AUTHORITY TO BIND SUCH ENTITY AND ITS AFFILIATES TO THESE TERMS AND CONDITIONS, IN WHICH CASE THE TERMS “YOU“ OR “YOUR“ SHALL REFER TO SUCH ENTITY AND ITS AFFILIATES. IF YOU DO NOT HAVE SUCH AUTHORITY, OR IF YOU DO NOT AGREE WITH THESE TERMS AND CONDITIONS, YOU MUST NOT ACCEPT THIS AGREEMENT AND MAY NOT USE THE SERVICES. 

1. DEFINITIONS

“Affiliate“ means any entity which directly or indirectly controls, is controlled by, or is under common control with the subject entity. “Control,“ for purposes of this definition, means direct or indirect ownership or control of more than 50% of the voting interests of the subject entity.

“Documentation” means User Guide, quick guide, FAQ, knowledge base articles, release notes or other documentation specific to the Products.

“Malicious Code“ means viruses, worms, time bombs, Trojan horses and other harmful or malicious code, files, scripts, agents or programs.

“Order Form“ means the ordering documents for purchases hereunder, including addenda thereto, that are entered into between You and Netop Business Solutions from time to time. 

“Products” means on-line or offline software products provided by Netop Business Solutions at www.netop.com or by Netop Business Solutions’ Affiliates at their websites or stores as subscriptions.

“Users“ means individuals who are authorized by You to use the Products, for whom subscriptions to a Product have been purchased, and who have been supplied with a specific and unique identification and password by You or by Netop Business Solutions identifying the specific user. Users may include but are not limited to Your employees, consultants, contractors and agents; or third parties with which You transact business.

2. USE OF FREE TRIAL

(a) Use of free trial: Netop Business Solutions makes Products available to You on a trial basis free of charge until the (a) the fourteenth (14) or the thirtieth (30) day after Your acceptance of this Agreement or (b) the start date of any purchased Products ordered by You. You are informed about additional trial terms and conditions at the trial registration web page. The free trial period may vary depending the type or version of the Products. You will be informed about the length of the trial at download. Any such additional terms and conditions are incorporated into this Agreement by reference and are legally binding. 

ANY DATA YOU ENTER INTO THE PRODUCTS, AND ANY CUSTOMIZATIONS MADE TO THE PRODUCTS BY OR FOR YOU, DURING YOUR FREE TRIAL MAY BE LOST UNLESS YOU PURCHASE A SUBSCRIPTION TO THE SAME PRODUCTS AS THOSE COVERED BY THE TRIAL, PURCHASE UPGRADED PRODUCTS, OR EXPORT SUCH DATA, BEFORE THE END OF THE FREE TRIAL PERIOD. YOU CANNOT TRANSFER DATA ENTERED OR CUSTOMIZATIONS MADE DURING THE FREE TRIAL TO A PRODUCT THAT WOULD BE A DOWNGRADE FROM THAT COVERED BY THE TRIAL (E.G. BASIC TO FREE VERSION) THEREFORE, IF YOU PURCHASE A SERVICE THAT WOULD BE A DOWNGRADE FROM THAT COVERED BY THE TRIAL, YOU MUST EXPORT YOUR DATA BEFORE THE END OF THE FREE TRIAL PERIOD OR YOUR DATA RISK BEING PERMANENTLY LOST. 

NOTWITHSTANDING SECTION 7 (WARRANTY AND DISCLAIMERS), DURING THE FREE TRIAL THE SERVICES ARE PROVIDED “AS-IS” AND “AS AVAILABLE” WITHOUT ANY WARRANTY. 

(b) Use of special versions: Some Products will at the end of a free trial period where no purchase was made or at the end of a subscription term where no renewal was purchased turn into a free version with limited functionality. You will be informed about free versions of the Products at download. Free versions can be used free of charge. Terms and conditions for use of free versions of the Products are incorporated into this Agreement by reference and are legally binding. 

3. USE OF THE PRODUCTS

3.1. Netop Business Solutions’ Responsibility. Netop Business Solutions or at Netop Business Solutions sole choice, Distributors and Partners appointed by Netop Business Solutions, shall provide  You basic support for the purchased Products at no additional charge according to Netop Business Solutions standard Service Level Agreement which can be found at www.netop.com. Netop Business Solutions will notify You of updates or new versions of the Products and any download of an updated version or new versions of the Products confirms your acceptance of this version. Information of new product updates and content will be available at www.netop.com and in the Documentation to the applicable product. Unless otherwise specified in the Order Form or Documentation, updates are included in the purchased Products.

3.2. Your Responsibilities. You shall (a) be responsible for Users’ compliance with this Agreement, (b) be solely responsible for the accuracy, quality, integrity and legality of Your Data and of the means by which You acquired Your Data, (c) use commercially reasonable efforts to prevent unauthorized access to or use of the Products, and notify Netop Business Solutions promptly of any such unauthorized access or use, and (iv) use the Products only in accordance with the User Guide or applicable Documentation and in accordance with any applicable law and/or government regulations. You shall not (a) make the Products available to anyone other than Users, (b) sell, resell, rent or lease the Products, (c) use the Products to store or transmit infringing, libelous, or otherwise unlawful or tortious material, or to store or transmit material in violation of third-party privacy rights, (d) use the Products to store or transmit Malicious Code, (e) interfere with or disrupt the integrity or performance of the Products or third-party data contained therein, (f) attempt to gain unauthorized access to the Products or their related systems or networks, or (g) use the Product to monitor, supervise, spy or log personal data if such usage is in violation with privacy rights or local laws. You or Your Users shall be responsible for complying with all domestic and international laws and any other any legal obligations, including but not limited to, obligations imposed by copyright, trademark, trade secrets, defamation, decency, privacy, security and export laws. 

3.3. Usage Limitations. Products may be subject to other limitations, such as, for example, limits on disk storage space, bandwidth, platform support or similar. Any such limitations are specified in the product Documentation. 

3.4. Rights to data. All data originating from use of the products saved in a memory, hard disk or any other subject are wholly owned by You as user or Customer to Netop Business Solutions. Netop Business Solutions is not liable for any content of such information. 

3.5. Storage of data. Netop Business Solutions will storage all information as one of the following:

1.  Data will be stored for the duration of this agreement governing the purchase of a service on Netop Business Solutions servers.
2.  Data will be stored for a fixed period as stated in the Documentation of the service. That means that data that exceeds that fixed period will be automatically removed. This is done in a chronological order – older data is removed and more recent is maintained.

At the termination date Netop Business Solutions will store all data (or all remaining data) for thirty (30) days after which the Data automatically will be deleted on Netop Business Solutions server. You are solely responsible for the retrieval and storage of all Data owned by You as part of Your use of the Service provided by Netop Business Solutions, and under no circumstance can Netop Business Solutions be held responsible for deleted data owned by You after the 30 days has expired from the date of termination.  

4. PURCHASE AND PAYMENT TERMS

4.1. Provision of Purchased Products. Netop Business Solutions shall make the purchased Products available to You pursuant to this Agreement and the relevant Order Forms during the subscription term. You agree that Your purchases hereunder are neither contingent on the delivery of any future functionality or features nor dependent on any oral or written public comments made by Netop Business Solutions regarding future functionality or features.

4.2. User Subscriptions. Unless otherwise specified in the applicable Order Form, Products are purchased as User subscriptions and may be accessed by no more than the specified number of Users. The user identification and password can only be used by one specific user and cannot be transferred or used by other users. Additional User subscriptions may be added at any time. User subscriptions are for designated Users and cannot be shared or used by more than one User at a time but may be reassigned to new Users replacing former Users who no longer require ongoing use of the Products.

4.3. Concurrent Subscriptions. The Concurrent Subscription gives You the right to install the associated Software on an unlimited number of Client Devices. The Software may be used on as many simultaneous sessions according to the number of subscriptions purchased. The Software is “in use“ on a Client Device when it is loaded into the temporary memory (i.e., random-access memory or RAM) or installed into the permanent memory (e.g., hard disk, CD-ROM, or other storage device) of that Client Device.

4.4. User Fees. You shall pay all fees specified in all Order Forms hereunder. Except as otherwise specified herein or in an Order Form, (a) fees are based on Products purchased or actual usage of Products, (b) payment obligations are non-cancelable and fees paid are non-refundable, and (c) the number of User subscriptions purchased cannot be decreased during the relevant subscription term stated on the Order Form. User subscription fees are based on monthly or annual payments that begin on the subscription start date and each monthly anniversary thereof or each twelve (12) months.

4.5. Invoicing and Payment. You will provide Netop Business Solutions with valid and updated credit card information or with a valid purchase order or alternative document reasonably acceptable to Netop Business Solutions.  If You provide credit card information to Netop Business Solutions, You authorize Netop Business Solutions to charge such credit for all Products and/or Services listed in the Order Form for the initial subscription term and any renewal subscription term(s).  Such charges shall be made in advance, either annually or in accordance with any different billing frequency stated in the applicable Order Form. If the Order Form specifies that payment will be by a method other than a credit card, Netop Business Solutions will invoice You in advance and otherwise in accordance with the relevant Order Form. Unless otherwise stated in the Order Form, invoiced charges are due net 30 days from the invoice date. You are responsible for maintaining complete and accurate billing and contact information to Netop Business Solutions.

4.6. Overdue Charges. If any charges are not received from You by the due date, then at Netop Business Solutions discretion, (a) such charges may accrue late interest at the rate of 1.5% of the outstanding balance per month, or the maximum rate permitted by law, whichever is lower, from the date such payment was due until the date paid, and/or (b) Netop Business Solutions may condition future subscription renewals and Order Forms on payment terms shorter than those specified in Section 4.5 (Invoicing and Payment).

4.7. Suspension of Service and Acceleration. If any amount owing by You under this or any other agreement for Netop Business Solutions services is 30 or more days overdue, Netop Business Solutions may, without limiting other rights and remedies, accelerate Your unpaid fee obligations under such agreements so that all such obligations become immediately due and payable, and suspend Your Product usage until such amounts are paid in full.

4.8. Taxes. Unless otherwise stated, Netop Business Solutions’ fees do not include any taxes, levies, duties or similar governmental assessments of any nature, including but not limited to value-added, sales, use or withholding taxes, assessable by any local, state, provincial, federal or foreign jurisdiction (collectively, “ Taxes“). You are responsible for paying all Taxes associated with Your purchases hereunder. If Netop Business Solutions has the legal obligation to pay or collect Taxes for which You are responsible under this paragraph, the appropriate amount shall be invoiced to and paid by You, unless You provide Netop Business Solutions with a valid tax exemption certificate authorized by the appropriate taxing authority. For clarity, Netop Business Solutions is solely responsible for taxes assessable against it based on Netop Business Solutions’ income, property and employees.

5. Service & Upgrade: Products purchased as subscriptions includes free product service, support and software upgrade(s). Please refer to www.netop.com/support-overview for further details about Netop’s support services.
 
Appendix C: Products provided as hosted services

1. Use of hosted Services. Netop Business Solutions provides hosted solutions and the software enabling the use thereof and related updates, including any documentation files accompanying the software (the software, any modifications or updates and documentation hereinafter collectively referred to as the “Software“). Software and Service are provided by Netop Business Solutions and/or its affiliates and assigned to You either as a Service hosted by Netop Business Solutions or by You. From time to time Netop Business Solutions will notify You of updates or new versions of the Software or Service. Your download of the updated or new versions of the Software and/or subsequent use of the Service confirms your acceptance of, and is also subject to, these Terms and Conditions. If you do not agree to abide by these Terms and Conditions, you may not use the Service or Software

2. General conditions and restrictions. Your responsibilities in connection with Services hosted by Netop Business Solutions: Netop Business Solutions strives to protect personal information and privacy, however, Netop Business Solutions cannot guarantee the security of any information You disclose online. You acknowledge by entering into this Agreement that you are aware of our limitations with respect to security and privacy. You agree to provide true, accurate, current and complete information as required by Netop Business Solutions. Unless produced by Netop Business Solutions, the information You access or receive by using the Software, and/or the Service, and information sent to you by visitors to your website, is provided by the visitors and is not, controlled, verified or endorsed by Netop Business Solutions in any way. 

By using the Software and the Service you agree to: (a) determine whether the information complies with Your needs; (b) determine whether You have adequate legal rights to store, reproduce or otherwise make use of the information in the manner contemplated by you; and (c) comply with all domestic and international laws and any other any legal obligations, including but not limited to, obligations imposed by copyright, trademark, trade secrets, defamation, decency, privacy, security and export laws. 
You understand and acknowledge that Netop Business Solutions has no obligation to monitor any portion of the Service, including without limitation, content posted by You or Your visitors. You acknowledge and agree, however, that Netop Business Solutions retains the right to monitor the Service, and to disclose any information as necessary or appropriate to satisfy any law, regulation or other governmental request, to operate the Service properly, or to protect the Service. Netop Business Solutions reserves the right to refuse to transmit or to remove any information or materials, in whole or in part, that, in Netop Business Solutions’ sole discretion, Netop Business Solutions deems unacceptable, undesirable, inappropriate or in violation of any law or these Terms and Conditions. You agree to immediately notify us of any unauthorized use of the Service or the Software or any other breach of security known to or suspected by you. Netop Business Solutions assumes no responsibility for the deletion of or failure to store electronic messages, chat communications, or other information submitted by You or Your visitors using the Software or Service, whether submitted as part of the Service or otherwise. 

If You violate these Terms and Conditions, You may be permanently or temporarily banned from using the Service and Software, at Netop Business Solutions’ sole discretion. By way of example but not limitation, Netop Business Solutions reserves the right to terminate this Agreement at any time if Netop Business Solutions determines, in its sole discretion, that Your site or activities are unsuitable for the Service. “Unsuitable“ sites or activities may include, without limitation, those sites that Netop Business Solutions believes: (a) display or promote obscene, indecent, pornographic or sexually explicit activities; (b) promote abusive, threatening, hateful, harmful, harassing, libelous or violent behavior; (c) promote discrimination based on race, sex, religion, nationality, disability, sexual orientation, or age; (d) promote illegal activities or fail to comply with applicable law; (e) violate the intellectual property rights of third parties; or (f) have the potential to interfere with the use, enjoyment, deployment or operation of the Service by other visitors.

3. Rights to data. All data originating from use of the products saved in a memory, hard disk or any other subject are wholly owned by You as user or Customer to Netop Business Solutions. Netop Business Solutions is not liable for any content of such information. 

4. Storage data. Netop Business Solutions will store all information as one of the following:

1.  Data will be stored for the duration of this agreement regulating the purchase of a service on Netop Business Solutions servers.
2.  Data will be stored for a fixed period as stated in the documentation of the service. That means that data that exceeds that fixed period will be automatically removed. This is done in a chronological order – older data is removed and more recent is maintained.

At the termination date Netop Business Solutions will storage all data for 30 days after which the Data automatically will be deleted on Netop Business Solutions server. You are sole responsible for the retrieval and storage of all Data owned by You as part of your use of the Service provided by Netop Business Solutions, and under no circumstance can Netop Business Solutions be held responsible for deleted data owned by You after the 30 days has expired from the date of termination.
  
5. Service & Upgrade: Products purchased as subscriptions include free product service, support and software upgrade(s). Please refer to www.netop.com/support-overview for further details about Netop’s support services.
 
Appendix D: Products purchased from App Stores

1. Acknowledgement. Netop Business Solutions and You acknowledge that this Agreement is between Netop Business Solutions and You only and not with Apple, Inc. (“Apple”) or Google Inc. (“Google”). Netop Business Solutions, not Apple or Google, is solely responsible for the Software and the content thereof. 

2. Scope of License. The license granted to the Software is limited to a non-transferable license to use the Software on any iPad, iPod Touch, iPhone and/or Android tablet that You own or control and as permitted by the Usage Rules set forth in the App Store Terms of Service – http://www.apple.com/legal/itunes/us/terms or in the Google Play Terms of Service – http://play.google.com/intl/en/about/play-terms.html

3. Maintenance and Support. Netop Business Solutions is solely responsible for providing any maintenance and support services with respect to the Software, as specified in the Agreement, or as required under applicable law. Netop Business Solutions and You acknowledge that Apple or Google has no obligation whatsoever to furnish any maintenance and support services with respect to the Software.

4. Warranty. Netop Business Solutions is solely responsible for any product warranties, whether express or implied by law, to the extent not effectively disclaimed. In the event of failure of the Software to conform to any applicable warranty, You may notify Apple or Google, and Apple or Google will refund the purchase price for the Software to You; and that, to the maximum extent permitted by applicable law, Apple or Google will have no other warranty obligation whatsoever with respect to the Software, and any other claims, losses, liabilities, damages, costs or expenses attributable to any failure to conform to any warranty will be Netop Business Solutions’ sole responsibility.

5. Product Claims. Netop Business Solutions and You acknowledge that Netop Business Solutions, not Apple or Google, is responsible for addressing any claims of You or any third party relating to the Software and Your possession and/or use of that Software, including, but not limited to: (i) product liability claims; (ii) any claim that the Software fails to conform to any applicable legal or regulatory requirement; and (iii) claims arising under consumer protection or similar legislation. 

6. Intellectual Property Rights. Netop Business Solutions and You acknowledge that, in the event of any third party claim that the Software or Your possession and use of that Software infringes that third party’s intellectual property rights, Netop Business Solutions, not Apple or Google, will be solely responsible for the investigation, defense, settlement and discharge of any such intellectual property infringement claim.

7. Legal Compliance. You must represent and warrant that (i) he/she is not located in a country that is subject to a U.S. Government embargo, or that has been designated by the U.S. Government as a “terrorist supporting” country; and (ii) he/she is not listed on any U.S. Government list of prohibited or restricted parties.

8. Third Party Beneficiary. Netop Business Solutions and You must acknowledge and agree that Apple,  Apple’s subsidiaries, Google and Google’s subsidiaries, are third party beneficiaries of the Agreement, and that, upon Your acceptance of the terms and conditions of the Agreement, Apple or Google will have the right (and will be deemed to have accepted the right) to enforce the Agreement against You as a third party beneficiary thereof.

9. Contact Information. Should you have any questions, complaints or claims with respect to the Agreement, please contact us at Netop Business Solutions A/S, Bregnerodvej 127, DK-3460 Birkerod, Denmark or +45 4590 25 25 or info@netop.com. 

10. Vendor signing. For the Android version of Netop Mobile and Embedded it is explicitly prohibited for end-users or any other to vendor sign or cause vendor signing of this module and thereby enabling this module to reach necessary system resources needed for the viewing of the screen and injection of input to the devices, that it otherwise could not reach. Doing so immediately revokes the license to use this module.

